// Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer: Hannes Roest $
// $Authors: Hannes Roest $
// --------------------------------------------------------------------------

#pragma once

#include <OpenMS/KERNEL/StandardTypes.h>
#include <OpenMS/KERNEL/MSSpectrum.h>
#include <OpenMS/KERNEL/MSChromatogram.h>
#include <OpenMS/KERNEL/MSExperiment.h>

#include <OpenMS/OPENSWATHALGO/DATAACCESS/ISpectrumAccess.h>

#include <memory>

namespace OpenMS
{
  /**
   * @brief An implementation of the OpenSWATH Spectrum Access interface completely in memory
   * 
   * This implementation of the spectrum access interface ensures that all data
   * is held completely in memory and is quickly accessible. This class can be
   * generated by passing any object implementing the Spectrum Access interface
   * and guarantees to provide the same access to the raw data as the original
   * object with the added benefits (and downside) of keeping all data in system
   * memory.
   * 
   * A possible example
   * 
   * @code
   * OpenSwath::ISpectrumAccess * data_access;
   * fillData(data_access); // assume that data_access points to some data
   * OpenSwath::ISpectrumAccess * in_memory_data_access = SpectrumAccessOpenMSInMemory(data_access);
   * @endcode
   *
   * After executing this code, two variables exist: data_access which provides
   * access to the original data but does so in one of multiple ways which is
   * not transparent to the user. However, in_memory_data_access will provide
   * access to the same data but with the guarantee that the data is available
   * in memory and not read from the disk.
   *
  */
  class OPENMS_DLLAPI SpectrumAccessOpenMSInMemory :
    public OpenSwath::ISpectrumAccess
  {
public:
    typedef OpenMS::PeakMap MSExperimentType;
    typedef OpenMS::MSSpectrum MSSpectrumType;
    typedef OpenMS::MSChromatogram MSChromatogramType;

    /// Constructor
    explicit SpectrumAccessOpenMSInMemory(OpenSwath::ISpectrumAccess & origin);

    /// Destructor
    ~SpectrumAccessOpenMSInMemory() override;

    /// Copy constructor
    SpectrumAccessOpenMSInMemory(const SpectrumAccessOpenMSInMemory & rhs);

    /// Light clone operator (actual data will not get copied)
    std::shared_ptr<OpenSwath::ISpectrumAccess> lightClone() const override;

    OpenSwath::SpectrumPtr getSpectrumById(int id) override;

    OpenSwath::SpectrumMeta getSpectrumMetaById(int id) const override;

    std::vector<std::size_t> getSpectraByRT(double RT, double deltaRT) const override;

    size_t getNrSpectra() const override;

    OpenSwath::ChromatogramPtr getChromatogramById(int id) override;

    size_t getNrChromatograms() const override;

    std::string getChromatogramNativeID(int id) const override;

private:

    std::vector< OpenSwath::SpectrumPtr > spectra_;
    std::vector< OpenSwath::SpectrumMeta > spectra_meta_;

    std::vector< OpenSwath::ChromatogramPtr > chromatograms_;
    std::vector< std::string > chromatogram_ids_;

  };

} //end namespace OpenMS

