// Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer: Timo Sachsenberg $
// $Authors: Erhan Kenar, Holger Franken, Mohammed Alhigaylan $
// --------------------------------------------------------------------------

#pragma once

#include <OpenMS/CONCEPT/ProgressLogger.h>
#include <OpenMS/DATASTRUCTURES/DefaultParamHandler.h>
#include <OpenMS/KERNEL/MassTrace.h>
#include <OpenMS/KERNEL/MSExperiment.h>
#include <OpenMS/KERNEL/StandardTypes.h>

#include <boost/dynamic_bitset.hpp>

namespace OpenMS
{

    /**
      @brief A mass trace extraction method that gathers peaks similar in m/z and moving along retention time.

      Peaks of a @ref MSExperiment are sorted by their intensity and stored in a
      list of potential chromatographic apex positions. Only peaks that are above
      the noise threshold (user-defined) are analyzed and only peaks that are n
      times above this minimal threshold are considered as apices. This saves
      computational resources and decreases the noise in the resulting output.

      Starting with these, mass traces are extended in- and decreasingly in
      retention time. During this extension phase, the centroid m/z is computed
      on-line as an intensity-weighted mean of peaks.

      The extension phase ends when either the frequency of gathered peaks drops
      below a threshold (min_sample_rate, see @ref MassTraceDetection parameters)
      or when the number of missed scans exceeds a threshold
      (trace_termination_outliers, see @ref MassTraceDetection parameters).

      Finally, only mass traces that pass a filter (a certain minimal and maximal
      length as well as having the minimal sample rate criterion fulfilled) get
      added to the result.

      @htmlinclude OpenMS_MassTraceDetection.parameters

      @ingroup Quantitation
    */
    class OPENMS_DLLAPI MassTraceDetection :
            public DefaultParamHandler,
            public ProgressLogger
    {
    public:
        /// Trace termination criteria enum for performance optimization
        enum TraceTerminationCriterion
        {
            OUTLIER,        ///< Terminate when consecutive outliers exceed threshold
            SAMPLE_RATE     ///< Terminate when sample rate falls below threshold
        };

        /// Default constructor
        MassTraceDetection();

        /// Default destructor
        ~MassTraceDetection() override;

        /** @name Helper methods
        */

        /** @name Main computation methods
        */

        /// Main method of MassTraceDetection. Extracts mass traces of a @ref MSExperiment and gathers them into a vector container.
        void run(const PeakMap &, std::vector<MassTrace> &, const Size max_traces = 0);

        /// Invokes the run method (see above) on merely a subregion of a @ref MSExperiment map.
        void run(PeakMap::ConstAreaIterator & begin, PeakMap::ConstAreaIterator & end, std::vector<MassTrace> & found_masstraces);

        /// determine if meta array is available
        bool hasFwhmMz() const { return has_fwhm_mz_; }
        bool hasFwhmIm() const { return has_fwhm_im_; }
        bool hasCentroidIm() const { return has_centroid_im_; }

        /** @name Private methods and members
        */
    protected:
        void updateMembers_() override;
        /// allows for the iterative computation of intensity weighted of a mass trace's centroid m/z or ion mobility
        static void updateIterativeWeightedMean_(const double& added_value,
                                                const double& added_intensity,
                                                double& centroid_value,
                                                double& prev_counter,
                                                double& prev_denom);

    private:

        struct Apex
        {
          Apex(double intensity, Size scan_idx, Size peak_idx);
          double intensity;
          Size scan_idx;
          Size peak_idx;
        };

        /// Encapsulates peak finding logic for both up and down directions
        struct PeakCandidate
        {
          Size idx;
          double mz;
          double intensity;
          double im;
          bool found;
          
          PeakCandidate() : idx(0), mz(-1.0), intensity(-1.0), im(-1.0), found(false) {}
        };

        /// Encapsulates trace extension state
        struct TraceExtensionState
        {
          Size scan_counter;
          Size hitting_peak_count;
          Size consecutive_missed;
          bool active;
          
          TraceExtensionState() : scan_counter(0), hitting_peak_count(0), consecutive_missed(0), active(true) {}
        };

        /// The internal run method
        void run_(const std::vector<Apex>& chrom_apices,
                  const Size peak_count,
                  const PeakMap & work_exp,
                  const std::vector<Size>& spec_offsets,
                  std::vector<MassTrace> & found_masstraces,
                  const Size max_traces = 0);

        /// Internal helper to extract and validate metadata float array indices
        void getIMIndices_(const PeakMap& spectra,
                           int& fwhm_meta_idx, bool& has_fwhm_mz,
                           int& im_idx, bool& has_centroid_im,
                           int& im_fwhm_idx, bool& has_fwhm_im) const;

        /// Find the best matching peak in a spectrum considering m/z and optionally ion mobility
        PeakCandidate findBestPeak_(const MSSpectrum& spectrum,
                                    double centroid_mz,
                                    double ftl_sd,
                                    double centroid_im = -1.0) const;

        /// Check if peak candidate meets acceptance criteria
        bool isPeakAcceptable_(const PeakCandidate& candidate,
                              double centroid_mz,
                              double ftl_sd,
                              double centroid_im,
                              Size spectrum_idx,
                              const std::vector<Size>& spec_offsets,
                              const boost::dynamic_bitset<>& peak_visited) const;

        /// Process a single peak during trace extension
        void processPeak_(const PeakCandidate& candidate,
                         const MSSpectrum& spectrum,
                         std::list<PeakType>& current_trace,
                         std::vector<std::pair<Size, Size>>& gathered_idx,
                         std::vector<double>& fwhms_mz,
                         std::vector<double>& fwhms_im,
                         double& centroid_mz,
                         double& centroid_im,
                         double& prev_counter,
                         double& prev_denom,
                         double& prev_counter_im,
                         double& prev_denom_im,
                         double& ftl_sd,
                         double& intensity_so_far,
                         Size spectrum_idx,
                         bool is_upward_extension);


        /// Check if a mass trace meets quality criteria
        bool isTraceValid_(const std::list<PeakType>& trace,
                          Size total_scans_visited,
                          Size consecutive_missed_down,
                          Size consecutive_missed_up) const;

        // Metadata availability flags – set by getIMIndices_ and valid after run_()
        bool has_fwhm_mz_ = false;
        bool has_fwhm_im_ = false;
        bool has_centroid_im_ = false;

        // parameter stuff
        double mass_error_ppm_;
        double mass_error_da_;
        double noise_threshold_int_;
        double chrom_peak_snr_;
        double ion_mobility_tolerance_;
        MassTrace::MT_QUANTMETHOD quant_method_;

        TraceTerminationCriterion trace_termination_criterion_;
        Size trace_termination_outliers_;
        double min_sample_rate_;
        double min_trace_length_;
        double max_trace_length_;

        bool reestimate_mt_sd_;

        // Metadata array indices - set during run_()
        mutable int fwhm_meta_idx_ = -1;
        mutable int ion_mobility_idx_ = -1;
        mutable int im_fwhm_idx_ = -1;
    };
}
