// Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer: Chris Bielow $
// $Authors: Juliane Schmachtenberg $
// --------------------------------------------------------------------------

#pragma once

#include <OpenMS/DATASTRUCTURES/String.h>
#include <OpenMS/QC/QCBase.h>

namespace OpenMS
{
  class FeatureMap;
  class MSExperiment;
  class PeptideIdentification;

  /**
    @brief QC metric calculating (un)calibrated m/z error

    The metric sets m/z-values of the original experiment and the calculated reference m/z-values, uncalibrated m/z error (ppm)
    and calibrated m/z error (ppm) as metavalues of all PeptideIdentifications in a FeatureMap.

    For full functionality a PeakMap/MSExperiment with original m/z-values before m/z calibration generated by InternalCalibration has to be given.
    It's also possible to use this without an MzML File, but then only uncalibrated m/z error (ppm) will be reported.
    A FeatureMap after FDR is always required.

    **/
  class OPENMS_DLLAPI MzCalibration : public QCBase
  {
  public:
    /// Constructor
    MzCalibration();

    /// Destructor
    virtual ~MzCalibration() = default;

    /**
     * @brief Writes results as meta values to the PeptideIdentification of the given FeatureMap
     * @param features FeatureMap with m/z-values of PeptideIdentification after calibration, meta values are added here
     * @param exp PeakMap of the original experiment. Can be empty (i.e. not available).
     * @param map_to_spectrum Map to find index of spectrum given by meta value at PepID
     * @throws Exception::InvalidParameter PeptideID is missing meta value 'spectrum_reference'
     * @throws Exception::IllegalArgument Spectrum for a PepID has MSLevel of 1
     * @throws Exception::MissingInformation Meta value 'mz_raw' missing from MSExperiment
     **/
    void compute(FeatureMap& features, const MSExperiment& exp, const QCBase::SpectraMap& map_to_spectrum);

    /// define the required input files
    /// only FeatureXML after FDR is ultimately necessary
    Status requirements() const override;

    /// Returns the name of the metric.
    const String& getName() const override;

  private:
    /// calculate the m/z values and m/z errors and add them to the PeptideIdentification
    void addMzMetaValues_(PeptideIdentification& peptide_ID, const MSExperiment& exp, const QCBase::SpectraMap& map_to_spectrum);

    double mz_raw_;
    double mz_ref_;
    bool no_mzml_;
  };
} // namespace OpenMS
