// Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer: Kyowon Jeong, Jihyung Kim, Jaekwan Kim $
// $Authors: Kyowon Jeong, Jihyung Kim, Jaekwan Kim $
// --------------------------------------------------------------------------

#include <OpenMS/ANALYSIS/TOPDOWN/DeconvolvedSpectrum.h>
#include <OpenMS/ANALYSIS/TOPDOWN/FLASHDeconvAlgorithm.h>
#include <OpenMS/ANALYSIS/TOPDOWN/FLASHIda.h>
#include <OpenMS/ANALYSIS/TOPDOWN/MassFeatureTrace.h>
#include <OpenMS/ANALYSIS/TOPDOWN/PeakGroup.h>
#include <OpenMS/ANALYSIS/TOPDOWN/Qvalue.h>
#include <OpenMS/ANALYSIS/TOPDOWN/TopDownIsobaricQuantification.h>
#include <OpenMS/PROCESSING/SPECTRAMERGING/SpectraMerger.h>
#include <OpenMS/PROCESSING/FILTERING/ThresholdMower.h>
#include <OpenMS/METADATA/SpectrumLookup.h>
#include <OpenMS/MATH/STATISTICS/GaussFitter.h>

#ifdef _OPENMP
  #include <omp.h>
#endif

namespace OpenMS
{
inline const Size max_peak_count_for_centroid_ = 3e4;
inline const Size max_peak_count_for_profile_ = 1e5;

FLASHDeconvAlgorithm::FLASHDeconvAlgorithm(): DefaultParamHandler("FLASHDeconvAlgorithm"), ProgressLogger()
{
  // FLASHIda log file read somewhere. ida_log_file_ should be updated.
  defaults_.setValue("ida_log", "", "Provide the log file generated by FLASHIda (e.g., IDA*.log) for coupling with FLASHIda acquisition.");
  defaults_.setValue(
    "report_FDR", "false",
    "Generate q-values (approximate FDR) for deconvolved masses. Decoy masses are also reported which were used for FDR calculation. (Beta version)");
  defaults_.setValidStrings("report_FDR", {"true", "false"});

  defaults_.setValue("allowed_isotope_error", 0,
                     "Tolerance for isotope index errors when calculating FDR. For instance, setting a value of 2 permits the inclusion of up to 2 isotope errors as valid matches. Beta version.");
  defaults_.addTag("allowed_isotope_error", "advanced");

  defaults_.setValue("use_RNA_averagine", "false", "Use the RNA (nucleotide) averagine model for deconvolution.");
  defaults_.setValidStrings("use_RNA_averagine", {"true", "false"});
  defaults_.addTag("use_RNA_averagine", "advanced");

//  defaults_.setValue(
//    "precursor_MS1_window", 1,
//    "Number of MS1 spectra around each MS2 spectrum to search for precursor peaks when determining the MS2 precursors. For MS2 spectrum, the mass of precursor ion should be determined for better deconvolution and reliable identification. "
//    "If the mass of precursor ion is not found in the immediately preceding MS1 spectrum, previous or next MS1 spectra may be used instead. "
//    "This parameter determines up to how many MS1 spectra around each MS2 spectrum will be searched.");
//  defaults_.setMinInt("precursor_MS1_window", 1);
//  defaults_.addTag("precursor_MS1_window", "advanced");

  defaults_.setValue(
    "isolation_window", 5.0,
    "Specify the isolation window width for precursor determination. Used when this information is absent in the mzML file.");
  defaults_.addTag("isolation_window", "advanced");

  defaults_.setValue("merging_method", 0,
                     "Method for merging spectra before deconvolution. 0: No merging  1: Gaussian averaging per MS level, effective for Q-TOF datasets. For MSn (n > 1), only the spectra from the same precursor mass "
                     "(subject to tolerance set by SD:tol) are averaged. 2: Block merging, combining all spectra into one per MS level (e.g., for NativeMS datasets).");
  defaults_.setMinInt("merging_method", 0);
  defaults_.setMaxInt("merging_method", 2);

  defaults_.setValue("merging_min_ms_level", 1, "Min MS level for merging");
  defaults_.setValue("merging_max_ms_level", 2, "Max MS level for merging");
  defaults_.setMinInt("merging_min_ms_level", 1);
  defaults_.setMinInt("merging_max_ms_level", 1);

  auto sd_defaults = SpectralDeconvolution().getDefaults();
  sd_defaults.remove("allowed_isotope_error");
  defaults_.insert("SD:", sd_defaults);

  Param mf_defaults = MassFeatureTrace().getDefaults();
  mf_defaults.setValue("min_cos", -1.0,
                       "Cosine similarity threshold between avg. and observed isotope pattern. When negative, MS1 cosine threshold for spectral "
                       "deconvolution (set by -SD:min_cos will be used ");
  mf_defaults.setValue(
    "mass_error_ppm", -1.0,
    "Specifies the mass error tolerance for feature tracing in ppm. When negative, the MS1 tolerance for deconvolution is used (e.g., 16 ppm is used when -SD:tol 16).");
  mf_defaults.addTag("min_cos", "advanced");
  mf_defaults.addTag("mass_error_ppm", "advanced");

  mf_defaults.remove("noise_threshold_int");
  mf_defaults.remove("reestimate_mt_sd");
  mf_defaults.remove("trace_termination_criterion");
  mf_defaults.remove("trace_termination_outliers");
  mf_defaults.remove("chrom_peak_snr");

  defaults_.insert("ft:", mf_defaults);
  defaults_.insert("iq:", TopDownIsobaricQuantification().getDefaults());

  defaultsToParam_();
}

void FLASHDeconvAlgorithm::updateMembers_()
{
  tols_ = param_.getValue("SD:tol");
  min_cos_ = param_.getValue("SD:min_cos");
  //precursor_MS1_window_ = param_.getValue("precursor_MS1_window");
  use_RNA_averagine_ = param_.getValue("use_RNA_averagine") != "false";
  report_decoy_ = param_.getValue("report_FDR") != "false";

  merge_spec_ = param_.getValue("merging_method");
  isolation_window_size_ = param_.getValue("isolation_window");
  ida_log_file_ = param_.getValue("ida_log").toString();
  current_min_ms_level_ = max_ms_level_;
  current_max_ms_level_ = 0;
}

void FLASHDeconvAlgorithm::updateMSLevels_(MSExperiment& map)
{
  // read input dataset once to count spectra
  for (auto& it : map)
  {
    // if forced_ms_level > 0, force MS level of all spectra to 1.
    if (forced_ms_level_ > 0) { it.setMSLevel(forced_ms_level_); }

    if (it.empty()) { continue; }
    if (it.getMSLevel() > max_ms_level_) { continue; }

    uint ms_level = it.getMSLevel();
    current_max_ms_level_ = current_max_ms_level_ < ms_level ? ms_level : current_max_ms_level_;
    current_min_ms_level_ = current_min_ms_level_ > ms_level ? ms_level : current_min_ms_level_;
  }
  // Max MS Level is adjusted according to the input dataset
  current_max_ms_level_ = current_max_ms_level_ > max_ms_level_ ? max_ms_level_ : current_max_ms_level_;
}

void FLASHDeconvAlgorithm::filterLowPeaks_(MSExperiment& map)
{
  OPENMS_LOG_INFO << "Filtering low peaks in spectra ... ";
  ThresholdMower threshold_mower_filter;                         // threshold
  Param t_filter_param = threshold_mower_filter.getParameters(); //"threshold", .00001
  t_filter_param.setValue("threshold", 1e-6);
  threshold_mower_filter.setParameters(t_filter_param);
  threshold_mower_filter.filterPeakMap(map);

#pragma omp parallel for default(none), shared(map)
  for (int i = 0; i < (int) map.size(); i++)
  {
    auto& it = map[i];
    if (it.empty()) continue;
    Size count = it.getType(false) == SpectrumSettings::CENTROID ? max_peak_count_for_centroid_ : max_peak_count_for_profile_;
    it.sortByIntensity(true);
    double threshold = it.size() < count ? 0 : it[count].getIntensity();
    threshold = std::max(threshold, (double)it.begin()->getIntensity() / 1000);
    // pop back the low intensity peaks using threshold

    while (! it.empty() && it.back().getIntensity() <= threshold)
    {
      it.pop_back();
    }
    it.sortByPosition();
  }
  OPENMS_LOG_INFO << "Done" << std::endl;
}

void FLASHDeconvAlgorithm::mergeSpectra_(MSExperiment& map, uint ms_level)
{
  SpectraMerger merger;
  merger.setLogType(CMD);
  Param sm_param = merger.getDefaults();
  sm_param.setValue("mz_binning_width", tols_[ms_level - 1] / 2.5);
  sm_param.setValue("mz_binning_width_unit", "ppm");
  uint min_ms_level = param_.getValue("merging_min_ms_level");
  uint max_ms_level = param_.getValue("merging_max_ms_level");
  if (merge_spec_ == 1 && ms_level >= min_ms_level && ms_level <= max_ms_level)
  {
    if (ms_level == 1)
    {
      OPENMS_LOG_INFO << "Gaussian averaging MS1 spectra ... " << std::endl;
      merger.setParameters(sm_param);
      map.sortSpectra();
      merger.average(map, "gaussian", (int)ms_level);
    }
    else
    {
      // For ms n, first find precursors for all ms n. then make a tmp map having the precursor masses as precursor
      std::map<String, std::vector<Precursor>> original_precursor_map;
//#pragma omp parallel for default(none), shared(map, ms_level, original_precursor_map)
      for (int i = 0; i < (int) map.size(); i++)
      {
        auto spec = map[i];
        if (spec.getMSLevel() != ms_level) continue;

        const auto& native_id = spec.getNativeID();
        original_precursor_map[native_id] = spec.getPrecursors();

        if (! spec.getPrecursors().empty() && native_id_precursor_peak_group_map_.find(native_id) != native_id_precursor_peak_group_map_.end())
        {
          auto precursor_pg = native_id_precursor_peak_group_map_[native_id];
          auto precursor = spec.getPrecursors()[0];
          precursor.setCharge(1);
          precursor.setMZ(precursor_pg.getMonoMass());
          precursor.setIntensity(precursor_pg.getIntensity());
          map[i].setPrecursors(std::vector<Precursor> {precursor});
        }
      }
      // merge MS n using precursor method
      OPENMS_LOG_INFO << "Merging MS" << ms_level << " spectra from the same deconvolved precursor masses... " << std::endl;
      sm_param.setValue("precursor_method:mz_tolerance", 0.2);
      sm_param.setValue("precursor_method:rt_tolerance", 30.0); // TODO make this as a user input?
      merger.setParameters(sm_param);
      map.sortSpectra();
      merger.mergeSpectraPrecursors(map);

      for (auto& mspec : map)
      {
        auto native_id_str = mspec.getNativeID();
        std::vector<String> native_ids;
        native_id_str.split(",", native_ids);

        for (auto& native_id : native_ids)
        {
          if (original_precursor_map.find(native_id) == original_precursor_map.end()) continue;
          mspec.setPrecursors(original_precursor_map[native_id]);
        }
      }
    }
  }
  else if (merge_spec_ == 2)
  {
    OPENMS_LOG_INFO << "Merging spectra into a single spectrum for MS" << ms_level << std::endl;
    sm_param.setValue("block_method:rt_block_size", map.size() + 1);
    map.sortSpectra();
    sm_param.setValue("block_method:ms_levels", IntList {(int)ms_level});
    merger.setParameters(sm_param);
    merger.mergeSpectraBlockWise(map);
  }
  filterLowPeaks_(map);
}

int FLASHDeconvAlgorithm::getScanNumber(const MSExperiment& map, Size index)
{
  auto native_id_str = map[index].getNativeID();
  std::vector<String> native_ids;
  native_id_str.split(",", native_ids);
  String type_accession = "MS:1000768";

  if (!map.getSourceFiles().empty())
  {
    type_accession = map.getSourceFiles()[0].getNativeIDTypeAccession();
    if (type_accession.empty()) type_accession = "MS:1000768";
  }
  int scan_number = SpectrumLookup::extractScanNumber(native_ids.back(), type_accession);
  if (scan_number < 0) { scan_number = (int)index + 1; }

  return scan_number;
}

std::vector<double> FLASHDeconvAlgorithm::getTolerances() const
{
  return tols_;
}

void FLASHDeconvAlgorithm::runSpectralDeconvolution_(MSExperiment& map, std::vector<DeconvolvedSpectrum>& deconvolved_spectra)
{
  startProgress(0, (SignedSize)map.size(), "running FLASHDeconv");
  std::map<double, int> rt_scan_map;
  for (Size index = 0; index < map.size(); index++)
  {
    int scan_number = getScanNumber(map, index);
    rt_scan_map[map[index].getRT()] = scan_number;
  }

  for (uint ms_level = 1; ms_level <= current_max_ms_level_; ms_level++)
  {
    if (ms_level > 1)
    {
      // here, register precursor peak groups to the ms2 spectra.
      findPrecursorPeakGroupsForMSnSpectra_(map, deconvolved_spectra, ms_level);
    }
    if (merge_spec_ > 0)
    {
      mergeSpectra_(map, ms_level);
      // repeat precursor peak group registration for merged/averaged spectra
      if (ms_level > 1 && merge_spec_ > 0)
      {
        // here, register precursor peak groups to the ms2 spectra.
        findPrecursorPeakGroupsForMSnSpectra_(map, deconvolved_spectra, ms_level);
      }
    }

    // run Spectral deconvolution
    for (Size index = 0; index < map.size(); index++)
    {
      int scan_number = merge_spec_ == 0? getScanNumber(map, index) :
                                         (rt_scan_map.find(map[index].getRT()) == rt_scan_map.end()? getScanNumber(map, index) :
                                                                                                    rt_scan_map[map[index].getRT()]);//getScanNumber(map, index);
      const auto& spec = map[index];

      if (ms_level != spec.getMSLevel()) { continue; }

      nextProgress();
      if (spec.empty()) { continue; }
      String native_id = spec.getNativeID();

      PeakGroup precursor_pg;
      if (native_id_precursor_peak_group_map_.find(native_id) != native_id_precursor_peak_group_map_.end())
        precursor_pg = native_id_precursor_peak_group_map_[native_id];

      // now do it
      sd_.performSpectrumDeconvolution(spec, scan_number, precursor_pg);

      auto& deconvolved_spectrum = sd_.getDeconvolvedSpectrum();

      // add precursor information
      if (native_id_precursor_peak_group_map_.find(native_id) != native_id_precursor_peak_group_map_.end())
      {
        deconvolved_spectrum.setDeconvolvedPrecursor(native_id_precursor_peak_map_[native_id]);
      }

      if (native_id_precursor_mass_intensity_map_.find(native_id) != native_id_precursor_mass_intensity_map_.end())
      {
        deconvolved_spectrum.setPrecursorMassIntensityMap(native_id_precursor_mass_intensity_map_[native_id]);
      }
      // add precursor scan number information when a precursor peak group is not found
      if (ms_level > 1 && precursor_pg.empty())
      {
        for (int p_index = (int)index - 1; p_index >= 0; p_index--)
        {
          if(map[p_index].getMSLevel() == ms_level - 1)
          {
            deconvolved_spectrum.setPrecursorScanNumber(getScanNumber(map, p_index));
            break;
          }
        }
      }

      if (report_decoy_ && !deconvolved_spectrum.empty())
      {
#pragma omp parallel sections default(none) shared(spec, scan_number, precursor_pg, deconvolved_spectrum)
        {
#pragma omp section
          sd_noise_decoy_.performSpectrumDeconvolution(spec, scan_number, precursor_pg);
#pragma omp section
          sd_signal_decoy_.performSpectrumDeconvolution(spec, scan_number, precursor_pg);
        }
        deconvolved_spectrum.sortByQscore();

        deconvolved_spectrum.reserve(deconvolved_spectrum.size() + sd_signal_decoy_.getDeconvolvedSpectrum().size()
                                     + sd_noise_decoy_.getDeconvolvedSpectrum().size());

        for (const auto& pg : sd_signal_decoy_.getDeconvolvedSpectrum())
            deconvolved_spectrum.push_back(pg);

        for (const auto& pg : sd_noise_decoy_.getDeconvolvedSpectrum())
            deconvolved_spectrum.push_back(pg);

        deconvolved_spectrum.sort();
      }
      deconvolved_spectra.push_back(deconvolved_spectrum);
    }
    std::sort(deconvolved_spectra.begin(), deconvolved_spectra.end());
  }
  endProgress();
}


const FLASHHelperClasses::PrecalculatedAveragine& FLASHDeconvAlgorithm::getAveragine()
{
  return sd_.getAveragine();
}

const FLASHHelperClasses::PrecalculatedAveragine& FLASHDeconvAlgorithm::getDecoyAveragine()
{
  if (!report_decoy_) return getAveragine();
  return sd_noise_decoy_.getAveragine();
}

std::vector<int> FLASHDeconvAlgorithm::getHistogram_(const std::vector<double>& data, double min_range, double max_range, double bin_size)
{
  int num_bins = static_cast<int>((max_range - min_range) / bin_size) + 1;
  std::vector<int> bins(num_bins, 0);

  // Populate the bins
  for (double value : data) {
    if (value >= min_range && value <= max_range) {
      int bin_index = static_cast<int>((value - min_range) / bin_size);
      bins[bin_index]++;
    }
  }
  return bins;
}

void FLASHDeconvAlgorithm::determineTolerance_(const MSExperiment& map, const Param& sd_param, const FLASHHelperClasses::PrecalculatedAveragine& avg, const uint ms_level)
{
  OPENMS_LOG_INFO << "Determining tolerance for MS" << ms_level << " ... ";
  auto sd = SpectralDeconvolution();
  auto sd_param_t = sd_param;
  sd.setAveragine(avg);
  tols_[ms_level - 1] = 200; // maximum tolerance
  sd_param_t.setValue("min_charge", 1); // better to include charge 1 to determine ppm error.
  sd_param_t.setValue("min_mass", 50.0); // better to include small masses to determine ppm error.
  sd_param_t.setValue("tol", tols_);
  sd.setParameters(sd_param_t);
  sd.setToleranceEstimation();

  int sample_rate = 100; //
  int count = 0;
  std::vector<double> sampled_tols;
  for (const auto& spec : map)
  {
    if (ms_level != (uint)spec.getMSLevel()) { continue; }
    if (spec.empty()) { continue; }
    if (count++ % sample_rate != 0) continue;
    PeakGroup precursor_pg;
    sd.performSpectrumDeconvolution(spec, 0, precursor_pg);

    const auto& deconvolved_spectrum = sd.getDeconvolvedSpectrum(); // estimate both.

    for (const auto& pg : deconvolved_spectrum)
    {
      if (pg.getQscore() < .9) // TODO automatically find the good mass threshold
        continue;
      for (auto error : pg.getMassErrors())
      {
        sampled_tols.push_back(error);
      }
    }
  }

  if (sampled_tols.size() < 6)
  {
    OPENMS_LOG_INFO << "failed. Cannot be determined - no MS" << ms_level << " spectrum. Set to 10ppm (default tolerance)." << std::endl;
    tols_[ms_level - 1] = 10;
    return;
  }

  try
  {
    const int l = -100, r = 100, bin_size = 1;
    const auto& bins = getHistogram_(sampled_tols, l, r, bin_size);
    // Calculate mean using std::accumulate

    std::vector<DPosition<2>> points;
    for (int b = l; b <= r; b += bin_size)
    {
      points.emplace_back(b, bins[b - l]);
    }
    Math::GaussFitter fitter;
    const auto fit = fitter.fit(points);

    tols_[ms_level - 1] = round(fit.sigma * 2.17 * 2); // 97% area under curve in gaussian
    OPENMS_LOG_INFO << "done. Determined tolerance: " << tols_[ms_level - 1] << " ppm. You may test around this tolerance for better results."
                    << std::endl;
  }
  catch (const Exception::UnableToFit& e)
  {
    // Handle the exception and issue a warning
    OPENMS_LOG_INFO << "failed. Cannot be determined - Gaussian fitting failure. Set to 10ppm (default tolerance)." << std::endl;
    tols_[ms_level - 1] = 10;
  }
  catch (const std::exception& e)
  {
    // Handle other exceptions if necessary
    OPENMS_LOG_INFO << "failed. Cannot be determined - Gaussian fitting failure. Set to 10ppm (default tolerance)." << std::endl;
    tols_[ms_level - 1] = 10;
  }
}

void FLASHDeconvAlgorithm::run(MSExperiment& map,
                               std::vector<DeconvolvedSpectrum>& deconvolved_spectra,
                               std::vector<FLASHHelperClasses::MassFeature>& deconvolved_features)
{
  // initialize
  precursor_map_for_ida_ = FLASHIda::parseFLASHIdaLog(ida_log_file_); // ms1 scan -> mass, charge ,score, mz range, precursor int, mass int, color

  updateMSLevels_(map);
  filterLowPeaks_(map);

  sd_ = SpectralDeconvolution();
  Param sd_param = param_.copy("SD:", true);
  sd_param.setValue("allowed_isotope_error", param_.getValue("allowed_isotope_error"));
  OPENMS_LOG_INFO<< "Calculating Averagines ... " << std::flush;
  sd_.setParameters(sd_param);
  sd_.calculateAveragine(use_RNA_averagine_);
  OPENMS_LOG_INFO<< "Done" << std::endl;
  const auto& avg = sd_.getAveragine();

  // determine tolerance in case tolerance input is negative
  for (uint ms_level = 1; ms_level <= current_max_ms_level_; ms_level++)
  {
    if (tols_[ms_level - 1] > 0) continue;
    determineTolerance_(map, sd_param, avg, ms_level);
  }

  sd_param.setValue("tol", tols_);
  sd_.setParameters(sd_param);

  if (report_decoy_)
  {
    sd_noise_decoy_.setParameters(sd_param);
    sd_noise_decoy_.setTargetDecoyType(PeakGroup::TargetDecoyType::noise_decoy, sd_.getDeconvolvedSpectrum()); // noise
    sd_noise_decoy_.calculateAveragine(use_RNA_averagine_); // for noise, averagine needs to be calculated differently.

    sd_signal_decoy_.setParameters(sd_param);
    sd_signal_decoy_.setTargetDecoyType(PeakGroup::TargetDecoyType::signal_decoy, sd_.getDeconvolvedSpectrum()); // isotope
    sd_signal_decoy_.setAveragine(avg);
  }

  setLogType(CMD);
  deconvolved_spectra.reserve(map.size() * 4);

  // run spectral deconvolution here and get deconvolved spectra
  runSpectralDeconvolution_(map, deconvolved_spectra);

  // feature tracing here and update FeatureQScores
  runFeatureFinding_(deconvolved_spectra, deconvolved_features);

  noise_decoy_weight_ = Qvalue::updatePeakGroupQvalues(deconvolved_spectra);

  TopDownIsobaricQuantification quantifier;
  Param quant_param = param_.copy("iq:", true);
  quantifier.setParameters(quant_param);
  // Isobaric quant run
  quantifier.quantify(map, deconvolved_spectra, deconvolved_features);
}

// currently only MS2 precursors
void FLASHDeconvAlgorithm::findPrecursorPeakGroupsFormIdaLog_(const MSExperiment& map, Size index, double start_mz, double end_mz)
{
  if (precursor_map_for_ida_.empty()) return;

  int scan_number = getScanNumber(map, index);
  auto filter_str = map[index].getMetaValue("filter string").toString();
  Size pos = filter_str.find("cv=");
  double cv = 1e5;

  if (pos != String::npos)
  {
    Size end = filter_str.find(" ", pos);
    if (end == String::npos) end = filter_str.length() - 1;
    cv = std::stod(filter_str.substr(pos + 3, end - pos));
  }

  for (auto iter = precursor_map_for_ida_.lower_bound(scan_number); iter != precursor_map_for_ida_.begin()
                                                                    && native_id_precursor_peak_group_map_.find(map[index].getNativeID()) == native_id_precursor_peak_group_map_.end(); iter--)
  {
    if (iter->first > scan_number || iter == precursor_map_for_ida_.end())
    {
      continue;
    }
    if (iter->first < scan_number - 50) // for FLASHIda, give more buffer scans
    {
      return;
    }

    for (auto& smap : iter->second)
    {
      if (abs(start_mz - smap[3]) < .001 && abs(end_mz - smap[4]) < .001)
      {
        FLASHHelperClasses::LogMzPeak precursor_log_mz_peak;
        precursor_log_mz_peak.abs_charge = std::abs((int)smap[1]);
        precursor_log_mz_peak.is_positive = (int)smap[1] > 0;
        precursor_log_mz_peak.isotopeIndex = 0;
        precursor_log_mz_peak.mass = smap[0];
        precursor_log_mz_peak.intensity = smap[6];

        PeakGroup precursor_pg(precursor_log_mz_peak.abs_charge, precursor_log_mz_peak.abs_charge, true);
        precursor_pg.push_back(precursor_log_mz_peak);
        precursor_pg.setAbsChargeRange(std::abs((int)smap[7]), std::abs((int)smap[8]));
        precursor_pg.setChargeIsotopeCosine(precursor_log_mz_peak.abs_charge, smap[9]);
        precursor_pg.setChargeSNR(precursor_log_mz_peak.abs_charge, smap[10]); // cnsr
        precursor_pg.setIsotopeCosine(smap[11]);
        precursor_pg.setSNR(smap[12]);
        precursor_pg.setChargeScore(smap[13]);
        precursor_pg.setAvgPPMError(smap[14]);
        precursor_pg.setQscore(smap[2]);
        precursor_pg.setRepAbsCharge(precursor_log_mz_peak.abs_charge);
        precursor_pg.updateMonoMassAndIsotopeIntensities(tols_[0]);
        int ms1_scan_number = iter->first;
        precursor_pg.setScanNumber(ms1_scan_number);
        Size index_copy (index);
        while(index_copy != 0 && getScanNumber(map, index_copy--) != ms1_scan_number);

        auto filter_str2 = map[index_copy].getMetaValue("filter string").toString(); // this part is messy. Make a function to parse CV from map
        Size pos2 = filter_str2.find("cv=");
        double cv_match = 1e5;

        if (pos2 != String::npos)
        {
          Size end2 = filter_str2.find(" ", pos2);
          if (end2 == String::npos) end2 = filter_str2.length() - 1;
          cv_match = std::stod(filter_str2.substr(pos2 + 3, end2 - pos2));
        }
        if (std::abs(cv_match - cv) > 1e-5) continue;

        native_id_precursor_peak_group_map_[map[index].getNativeID()] = precursor_pg;
        break;
      }
    }
  }
}

void FLASHDeconvAlgorithm::findPrecursorPeakGroupsForMSnSpectra_(const MSExperiment& map,
                                                                 const std::vector<DeconvolvedSpectrum>& deconvolved_spectra,
                                                                 uint ms_level)
{
  for (Size index = 0; index < map.size(); index++)
  {
    auto spec = map[index]; // MS2 index
    if (spec.getMSLevel() != ms_level) { continue; }

    int scan_number = getScanNumber(map, index);
    String native_id = spec.getNativeID();

    auto filter_str = spec.getMetaValue("filter string").toString();
    Size pos = filter_str.find("cv=");
    double cv = 1e5;

    if (pos != String::npos)
    {
      Size end = filter_str.find(" ", pos);
      if (end == String::npos) end = filter_str.length() - 1;
      cv = std::stod(filter_str.substr(pos + 3, end - pos));
    }
//
//    // find all candidate scan numbers from ms_level - 1
//    int num_precursor_window = ms_level == 2 ? precursor_MS1_window_ : 1;
//    auto index_copy = index;
//    while (index_copy > 0 && num_precursor_window > 0)
//    {
//      index_copy--;
//      if (map[index_copy].getMSLevel() == ms_level - 1) { num_precursor_window--; }
//    }
//
//    int b_scan_number = getScanNumber(map, index_copy);
//    index_copy = index;
//
//    num_precursor_window = ms_level == 2 ? precursor_MS1_window_ : 0;
//    while (index_copy < map.size() - 1 && num_precursor_window > 0)
//    {
//      index_copy++;
//      if (map[index_copy].getMSLevel() == ms_level - 1) { num_precursor_window--; }
//    }
//
//    //int a_scan_number = getScanNumber(map, index_copy);
//    // then find deconvolved spectra within the scan numbers.
//    auto biter = std::lower_bound(deconvolved_spectra.begin(), deconvolved_spectra.end(), DeconvolvedSpectrum(b_scan_number));
//    //auto aiter = std::lower_bound(deconvolved_spectra.begin(), deconvolved_spectra.end(), DeconvolvedSpectrum(a_scan_number));
//
//    std::vector<DeconvolvedSpectrum> survey_scans;
//
//    // cv mismatches
//    while (biter < deconvolved_spectra.end() && biter <= aiter)
//    {
//      if ((biter->getOriginalSpectrum().getMSLevel() == ms_level - 1) && (std::abs(biter->getCV() - cv) < 1e-5)) { survey_scans.push_back(*biter); }
//      biter++;
//    }
//
//    std::sort(survey_scans.begin(), survey_scans.end(), [scan_number](const DeconvolvedSpectrum& a, const DeconvolvedSpectrum& b) {
//      int da = std::abs(a.getScanNumber() - scan_number);
//      int db = std::abs(b.getScanNumber() - scan_number);
//
//      if (da != db) return da < db;
//
//      return a.getScanNumber() < b.getScanNumber();
//    });



    DeconvolvedSpectrum precursor_deconvolved_spectrum;
    MSSpectrum precursor_raw_spec;

    auto index_copy = index;
    while (index_copy > 0)
    {
      index_copy--;
      if (map[index_copy].getMSLevel() == ms_level - 1)
      {
        auto _filter_str = map[index_copy].getMetaValue("filter string").toString();
        Size _pos = _filter_str.find("cv=");
        double _cv = 1e5;

        if (_pos != String::npos)
        {
          Size _end = _filter_str.find(" ", _pos);
          if (_end == String::npos) _end = _filter_str.length() - 1;
          _cv = std::stod(_filter_str.substr(_pos + 3, _end - _pos));
        }
        if (cv == _cv)
        {
          precursor_raw_spec = map[index_copy];
          break;
        }
      }
    }

    if (precursor_raw_spec.empty()) continue;

    auto iter = std::lower_bound(deconvolved_spectra.begin(), deconvolved_spectra.end(), DeconvolvedSpectrum(scan_number));
    while (iter != deconvolved_spectra.begin())
    {
      if (std::abs(iter->getOriginalSpectrum().getRT() - precursor_raw_spec.getRT()) < 1e-5)
      {
        precursor_deconvolved_spectrum = *iter;
        break;
      }
      iter--;
    }

    // register the best precursor, starting from the most recent one. Out of the masses in a single scan, use the max SNR one.
    double start_mz = 0;
    double end_mz = 0;
    for (auto& precursor : spec.getPrecursors())
    {
      double loffset = precursor.getIsolationWindowLowerOffset();
      double uoffest = precursor.getIsolationWindowUpperOffset();
      loffset = loffset <= 0 ? isolation_window_size_ / 2.0 : loffset;
      uoffest = uoffest <= 0 ? isolation_window_size_ / 2.0 : uoffest;

      start_mz = loffset > 100.0 ? loffset : -loffset + precursor.getMZ();
      end_mz = uoffest > 100.0 ? uoffest : uoffest + precursor.getMZ();
    }

    double sum_intensity = .0;

    const Size k = precursor_raw_spec.findNearest(start_mz);
    for (Size l = k + 1; l < precursor_raw_spec.size(); l++)
    {
      if (precursor_raw_spec[l].getMZ() < start_mz) continue;
      if (precursor_raw_spec[l].getMZ() > end_mz) break;
      sum_intensity += precursor_raw_spec[l].getIntensity();
    }

    for (Size l = k; ; l--)
    {
      if (precursor_raw_spec[l].getMZ() < start_mz) break;
      if (precursor_raw_spec[l].getMZ() > end_mz) continue;
      sum_intensity += precursor_raw_spec[l].getIntensity();
      if (l == 0) break;
    }


    double max_snr = -1.0;
    //auto o_spec = precursor_deconvolved_spectrum.getOriginalSpectrum();
    std::map<double, double> mass_to_intensity;
    mass_to_intensity[.0] = sum_intensity; // total intensity

    for (auto& pg : precursor_deconvolved_spectrum)
    {
      if (pg[0].mz > end_mz || pg.back().mz < start_mz) { continue; }

      double max_intensity = .0;
      const FLASHHelperClasses::LogMzPeak* tmp_precursor = nullptr;

      int c = int(round(pg.getMonoMass() / start_mz));
      double intensity = 0;
      for (auto& tmp_peak : pg)
      {
        if (tmp_peak.abs_charge != c) { continue; }
        if (tmp_peak.mz < start_mz || tmp_peak.mz > end_mz) { continue; }
        intensity += tmp_peak.intensity;

        if (tmp_peak.intensity < max_intensity) { continue; }
        max_intensity = tmp_peak.intensity;
        tmp_precursor = &tmp_peak;
      }

      if (tmp_precursor == nullptr) { continue; }
      mass_to_intensity[pg.getMonoMass()] = intensity;

      auto snr= pg.getChargeSNR(tmp_precursor->abs_charge); // the highest snr one should determine the mass

      if (snr < max_snr) { continue; }
      max_snr = snr;

      native_id_precursor_peak_group_map_[native_id] = pg;
      Precursor precursor;
      precursor.setCharge(pg.getRepAbsCharge());
      precursor.setMZ(tmp_precursor->mz);
      precursor.setIntensity(tmp_precursor->intensity);

      native_id_precursor_peak_map_[native_id] = precursor;
    }

    for (const auto& [m, j] : mass_to_intensity)
    {
      native_id_precursor_mass_intensity_map_[native_id].emplace_back(m, j);
    }
    if (native_id_precursor_peak_group_map_.find(native_id) == native_id_precursor_peak_group_map_.end())
    {
      findPrecursorPeakGroupsFormIdaLog_(map, index, start_mz, end_mz);
    }
  }
}

void FLASHDeconvAlgorithm::updatePrecursorQScores_(std::vector<DeconvolvedSpectrum>& deconvolved_spectra, int ms_level)
{
  // update precursor feature QScores and qvalues
  std::map<int, DeconvolvedSpectrum> scan_fullscan;

  for (auto& dspec : deconvolved_spectra)
  {
    if ((int)dspec.getOriginalSpectrum().getMSLevel() != ms_level - 1) continue;
    int scan = dspec.getScanNumber();
    scan_fullscan[scan] = dspec;
  }

  for (auto& dspec : deconvolved_spectra)
  {
    if ((int)dspec.getOriginalSpectrum().getMSLevel() != ms_level) continue;
    if (dspec.getPrecursorPeakGroup().empty()) continue;

    auto precursor_pg = dspec.getPrecursorPeakGroup();

    int pscan = precursor_pg.getScanNumber();
    if (scan_fullscan.find(pscan) == scan_fullscan.end()) continue;

    auto fullscan = scan_fullscan[pscan];

    auto iter = std::lower_bound(fullscan.begin(), fullscan.end(), precursor_pg);

    if (iter == fullscan.end()) continue;

    if (precursor_pg.getMonoMass() == iter->getMonoMass())
    {
      precursor_pg.setFeatureIndex(iter->getFeatureIndex());
      precursor_pg.setQscore(iter->getQscore());
      if (iter->getFeatureIndex() > 0) precursor_pg.setQscore2D(iter->getQscore2D());
    }
    else { precursor_pg.setFeatureIndex(0); }
    dspec.setPrecursorPeakGroup(precursor_pg);
  }
}

void FLASHDeconvAlgorithm::runFeatureFinding_(std::vector<DeconvolvedSpectrum>& deconvolved_spectra,
                                              std::vector<FLASHHelperClasses::MassFeature>& deconvolved_features)
{
  if (merge_spec_ == 2) return;

  auto mass_tracer = MassFeatureTrace();
  auto decoy_mass_tracer = MassFeatureTrace();

  Param mf_param = param_.copy("ft:", true);

  if (((double)mf_param.getValue("mass_error_ppm")) < 0) { mf_param.setValue("mass_error_ppm", tols_[0]); }
  if (((double)mf_param.getValue("min_cos")) < 0) { mf_param.setValue("min_cos", min_cos_[0]); }

  mf_param.setValue("noise_threshold_int", .0);
  mf_param.setValue("reestimate_mt_sd", "false");
  mf_param.setValue("trace_termination_criterion", "outlier");
  mf_param.setValue("trace_termination_outliers", 20);
  mf_param.setValue("chrom_peak_snr", .0);

  mass_tracer.setParameters(mf_param); // maybe go to set param
  // Find features for MS1 or the minimum MS level in the dataset.
  deconvolved_features = mass_tracer.findFeaturesAndUpdateQscore2D(sd_.getAveragine(), deconvolved_spectra, (int)current_min_ms_level_, false);

  if (report_decoy_)
  {
    const auto& decoy_deconvolved_features = mass_tracer.findFeaturesAndUpdateQscore2D(sd_.getAveragine(), deconvolved_spectra, (int)current_min_ms_level_, true);
    deconvolved_features.insert(deconvolved_features.end(), decoy_deconvolved_features.begin(), decoy_deconvolved_features.end());
  }

  mf_param.setValue("min_trace_length", 1e-5); // allow all traces for MSn
  mass_tracer.setParameters(mf_param);
  // Find features for MSn
  for (int ms_level = (int)current_min_ms_level_ + 1; ms_level <= (int)current_max_ms_level_; ms_level++)
  {
    updatePrecursorQScores_(deconvolved_spectra, ms_level);
    std::map<uint, std::vector<Size>> feature_index_set;
    for (Size i = 0; i < deconvolved_spectra.size(); i++)
    {
      const auto& dspec = deconvolved_spectra[i];
      if ((int)dspec.getOriginalSpectrum().getMSLevel() != ms_level) continue;
      if (dspec.getPrecursorPeakGroup().empty()) continue;
      uint findex = dspec.getPrecursorPeakGroup().getFeatureIndex();
      if (findex == 0) continue;
      feature_index_set[findex].push_back(i);
    }

    for (const auto& element : feature_index_set)
    {
      std::vector<DeconvolvedSpectrum> tmp_dspec;
      tmp_dspec.reserve(element.second.size());
      for (Size i : element.second)
        tmp_dspec.push_back(deconvolved_spectra[i]);

      const auto& df = mass_tracer.findFeaturesAndUpdateQscore2D(sd_.getAveragine(), tmp_dspec, ms_level, false);
      deconvolved_features.insert(deconvolved_features.end(), df.begin(), df.end());

      if (report_decoy_)
      {
        const auto& df_decoy = mass_tracer.findFeaturesAndUpdateQscore2D(sd_.getAveragine(), tmp_dspec, ms_level, true);
        deconvolved_features.insert(deconvolved_features.end(), df_decoy.begin(), df_decoy.end());
      }

      Size j = 0;
      for (Size i : element.second)
        deconvolved_spectra[i] = tmp_dspec[j++];
    }
  }
}
} // namespace OpenMS