// Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer: $
// $Authors: Hendrik Weisser, Stephan Aiche $
// --------------------------------------------------------------------------

#include <OpenMS/CONCEPT/ClassTest.h>
#include <OpenMS/test_config.h>

///////////////////////////

#include <OpenMS/ANALYSIS/MAPMATCHING/TransformationModel.h>
#include <OpenMS/ANALYSIS/MAPMATCHING/TransformationModelInterpolated.h>

#include <OpenMS/FORMAT/CsvFile.h>

///////////////////////////

START_TEST(TransformationModelInterpolated, "$Id$")

/////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////

using namespace OpenMS;
using namespace std;

TransformationModelInterpolated* ptr = nullptr;
TransformationModelInterpolated* nullPointer = nullptr;

TransformationModel::DataPoints dummy_data;
dummy_data.push_back(make_pair(0.0, 1.0));
dummy_data.push_back(make_pair(0.5, 4.0));
dummy_data.push_back(make_pair(1.0, 2.0));
dummy_data.push_back(make_pair(1.0, 4.0));

START_SECTION((TransformationModelInterpolated(const DataPoints &data, const Param &params)))
{
  ptr = new TransformationModelInterpolated(dummy_data, Param());
  TEST_NOT_EQUAL(ptr, nullPointer)
}
END_SECTION

START_SECTION((~TransformationModelInterpolated()))
{
  delete ptr;
}
END_SECTION

START_SECTION((double evaluate(double value) const))
{
  // NOTE: This test ensure first of all the compatibility of the interpolation
  //       to the original GSL based implementation of TransformationModelInterpolated

  /* load data generated by gsl interpolation
   */
  CsvFile data_points(OPENMS_GET_TEST_DATA_PATH("TransformationModelInterpolated_base_data.txt"), '\t');
  TransformationModel::DataPoints base_data;
  for (Size i = 0; i < data_points.rowCount(); ++i)
  {
    StringList sl;
    if (data_points.getRow(i, sl))
    {
      base_data.push_back(make_pair(sl[0].toDouble(), sl[1].toDouble()));
    }
  }

  // create interpolations
  Param p_linear;
  TransformationModelInterpolated::getDefaultParameters(p_linear);
  p_linear.setValue("interpolation_type", "linear");
  TransformationModelInterpolated linear_interpolation(base_data, p_linear);

  Param p_cspline;
  TransformationModelInterpolated::getDefaultParameters(p_cspline);
  p_cspline.setValue("interpolation_type", "cspline");
  TransformationModelInterpolated cspline_interpolation(base_data, p_cspline);

  Param p_akima;
  TransformationModelInterpolated::getDefaultParameters(p_akima);
  p_akima.setValue("interpolation_type", "akima");
  TransformationModelInterpolated akima_interpolation(base_data, p_akima);


  // read gsl results
  CsvFile gsl_results(OPENMS_GET_TEST_DATA_PATH("TransformationModelInterpolated_gsl_data.txt"), '\t');
  std::vector<double> gsl_target_points;
  std::vector<double> gsl_cspline;
  std::vector<double> gsl_linear;
  std::vector<double> gsl_akima;

  for (Size i = 0; i < gsl_results.rowCount(); ++i)
  {
    StringList sl;
    if (gsl_results.getRow(i, sl))
    {
      gsl_target_points.push_back(sl[0].toDouble());
      gsl_linear.push_back(sl[1].toDouble());
      gsl_akima.push_back(sl[2].toDouble());
      gsl_cspline.push_back(sl[3].toDouble());
    }
  }

  // test the interpolation
  for (size_t i = 0; i < gsl_target_points.size(); ++i)
  {
    const double x = gsl_target_points[i];
    TEST_REAL_SIMILAR(linear_interpolation.evaluate(x), gsl_linear[i])
    TEST_REAL_SIMILAR(cspline_interpolation.evaluate(x), gsl_cspline[i])
    TEST_REAL_SIMILAR(akima_interpolation.evaluate(x), gsl_akima[i])
  }
}
END_SECTION

START_SECTION(([EXTRA] TransformationModelInterpolated::evaluate() beyond the actual borders))
{
  Param p;
  TransformationModelInterpolated::getDefaultParameters(p);

  TransformationModelInterpolated tr(dummy_data, p);
  TEST_REAL_SIMILAR(tr.evaluate(0.0), 1.0);
  TEST_REAL_SIMILAR(tr.evaluate(0.5), 4.0);
  TEST_REAL_SIMILAR(tr.evaluate(1.0), 3.0);

  // using default value, two-point-linear
  {
    // Without changing parameters, extrapolation is linear and independent of
    // the actual interpolation beyond the borders 
    TransformationModelInterpolated tr(dummy_data, Param());

    // see TransformationModelLinear_test
    TEST_REAL_SIMILAR(tr.evaluate(-0.5), 0.0);
    TEST_REAL_SIMILAR(tr.evaluate(1.5), 4.0);
  }

  {
    TransformationModelInterpolated::getDefaultParameters(p);
    p.setValue("extrapolation_type", "two-point-linear");
    TransformationModelInterpolated tr(dummy_data, p);
    TEST_REAL_SIMILAR(tr.evaluate(-0.5), 0.0); // slope of 2
    TEST_REAL_SIMILAR(tr.evaluate(1.5), 4.0); // slope of 2
  }

  {
    TransformationModelInterpolated::getDefaultParameters(p);
    p.setValue("extrapolation_type", "four-point-linear");
    TransformationModelInterpolated tr(dummy_data, p);
    TEST_REAL_SIMILAR(tr.evaluate(-0.5), -2.0); // slope of 6
    TEST_REAL_SIMILAR(tr.evaluate(1.5), 2.0); // slope of -2
  }

  {
    TransformationModelInterpolated::getDefaultParameters(p);
    p.setValue("extrapolation_type", "global-linear");
    TransformationModelInterpolated tr(dummy_data, p);
    TEST_REAL_SIMILAR(tr.evaluate(-0.5), 0.909091); // 1.727 + 1.636 * -0.5
    TEST_REAL_SIMILAR(tr.evaluate(1.5), 4.1818182); // 1.727 + 1.636 * 1.5
  }

}
END_SECTION

START_SECTION((static void getDefaultParameters(Param & params)))
{
  Param p;
  TransformationModelInterpolated::getDefaultParameters(p);
  TEST_EQUAL(!p.empty(), true)
  TEST_EQUAL(p.getValue("interpolation_type"), "cspline")
  TEST_EQUAL(p.getValue("extrapolation_type"), "two-point-linear")
}
END_SECTION

/////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////
END_TEST
