// Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer: Hendrik Weisser $
// $Authors: Marc Sturm, Clemens Groepl, Hendrik Weisser $
// --------------------------------------------------------------------------

#include <OpenMS/APPLICATIONS/MapAlignerBase.h>

#include <OpenMS/FORMAT/FileHandler.h>

using namespace OpenMS;
using namespace std;

//-------------------------------------------------------------
// Doxygen docu
//-------------------------------------------------------------

/**
@page TOPP_MapRTTransformer MapRTTransformer

@brief Applies retention time transformations to maps.

<CENTER>
    <table>
        <tr>
            <th ALIGN = "center"> potential predecessor tools </td>
            <td VALIGN="middle" ROWSPAN=2> &rarr; MapRTTransformer &rarr;</td>
            <th ALIGN = "center"> potential successor tools </td>
        </tr>
        <tr>
            <td VALIGN="middle" ALIGN = "center" ROWSPAN=1> @ref TOPP_MapAlignerIdentification @n (or another alignment algorithm) </td>
            <td VALIGN="middle" ALIGN = "center" ROWSPAN=1> @ref TOPP_FeatureLinkerUnlabeled or @n @ref TOPP_FeatureLinkerUnlabeledQT </td>
        </tr>
    </table>
</CENTER>

This tool can apply retention time transformations to different types of data (mzML, featureXML, consensusXML, and idXML files).
The transformations might have been generated by a previous invocation of one of the MapAligner tools (linked below).
However, the trafoXML file format is not very complicated, so it is relatively easy to write (or generate) your own files.
Each input file will give rise to one output file.

@see @ref TOPP_MapAlignerIdentification @ref TOPP_MapAlignerPoseClustering

With this tool it is also possible to invert transformations, or to fit a different model than originally specified to the retention time data in the transformation files. To fit a new model, choose a value other than "none" for the model type (see below).

Original retention time values can be kept as meta data. With the option @p store_original_rt, meta values with the name "original_RT" and the original retention time will be created for every major data element (spectrum, chromatogram, feature, consensus feature, peptide identification), unless they already exist - "original_RT" values from a previous invocation will not be overwritten.

Since %OpenMS 1.8, the extraction of data for the alignment has been separate from the modeling of RT transformations based on that data. It is now possible to use different models independently of the chosen algorithm. The different available models are:
- @ref OpenMS::TransformationModelLinear "linear": Linear model.
- @ref OpenMS::TransformationModelBSpline "b_spline": Smoothing spline (non-linear).
- @ref OpenMS::TransformationModelInterpolated "interpolated": Different types of interpolation.

The following parameters control the modeling of RT transformations (they can be set in the "model" section of the INI file):
@htmlinclude OpenMS_MapRTTransformerModel.parameters @n

@note As output options, either @p out or @p trafo_out has to be provided. They can be used together.

@note Currently mzIdentML (mzid) is not directly supported as an input/output format of this tool. Convert mzid files to/from idXML using @ref TOPP_IDFileConverter if necessary.

<B>The command line parameters of this tool are:</B> @n
@verbinclude TOPP_MapRTTransformer.cli
<B>INI file documentation of this tool:</B>
@htmlinclude TOPP_MapRTTransformer.html

*/

// We do not want this class to show up in the docu:
/// @cond TOPPCLASSES

class TOPPMapRTTransformer :
  public TOPPBase
{

public:
  TOPPMapRTTransformer() :
    TOPPBase("MapRTTransformer", "Applies retention time transformations to maps.")
  {
  }

protected:
  void registerOptionsAndFlags_() override
  {
    String file_formats = "mzML,featureXML,consensusXML,idXML";
    // "in" is not required, in case we only want to invert a transformation:
    registerInputFile_("in", "<file>", "", "Input file to transform (separated by blanks)", false);
    setValidFormats_("in", ListUtils::create<String>(file_formats));
    registerOutputFile_("out", "<file>", "", "Output file (same file type as 'in'). This option or 'trafo_out' has to be provided; they can be used together.", false);
    setValidFormats_("out", ListUtils::create<String>(file_formats));
    registerInputFile_("trafo_in", "<file>", "", "Transformation to apply");
    setValidFormats_("trafo_in", ListUtils::create<String>("trafoXML"));
    registerOutputFile_("trafo_out", "<file>", "", "Transformation output file. This option or 'out' has to be provided; they can be used together.", false);
    setValidFormats_("trafo_out", ListUtils::create<String>("trafoXML"));
    registerFlag_("invert", "Invert transformation (approximatively) before applying it");
    registerFlag_("store_original_rt", "Store the original retention times (before transformation) as meta data in the output file");
    addEmptyLine_();

    registerSubsection_("model", "Options to control the modeling of retention time transformations from data");
  }

  Param getSubsectionDefaults_(const String& /* section */) const override
  {
    return MapAlignerBase::getModelDefaults("none");
  }

  template <class TMap>
  void applyTransformation_(const TransformationDescription& trafo, TMap& map)
  {
    bool store_original_rt = getFlag_("store_original_rt");
    MapAlignmentTransformer::transformRetentionTimes(map, trafo,
                                                     store_original_rt);
    addDataProcessing_(map, getProcessingInfo_(DataProcessing::ALIGNMENT));
  }

  ExitCodes main_(int, const char**) override
  {
    //-------------------------------------------------------------
    // parameter handling
    //-------------------------------------------------------------
    String in = getStringOption_("in");
    String out = getStringOption_("out");
    String trafo_in = getStringOption_("trafo_in");
    String trafo_out = getStringOption_("trafo_out");
    Param model_params = getParam_().copy("model:", true);
    String model_type = model_params.getValue("type").toString();
    model_params = model_params.copy(model_type + ":", true);

    ProgressLogger progresslogger;
    progresslogger.setLogType(log_type_);

    //-------------------------------------------------------------
    // check for valid input
    //-------------------------------------------------------------
    if (out.empty() && trafo_out.empty())
    {
      writeLogError_("Error: A data or a transformation output file has to be provided (parameters 'out'/'trafo_out')");
      return ILLEGAL_PARAMETERS;
    }
    if (in.empty() != out.empty())
    {
      writeLogError_("Error: Data input and output parameters ('in'/'out') must be used together");
      return ILLEGAL_PARAMETERS;
    }

    //-------------------------------------------------------------
    // apply transformation
    //-------------------------------------------------------------
    TransformationDescription trafo;
    FileHandler().loadTransformations(trafo_in, trafo, true, {FileTypes::TRANSFORMATIONXML});
    if (model_type != "none")
    {
      trafo.fitModel(model_type, model_params);
    }
    if (getFlag_("invert"))
    {
      trafo.invert();
    }
    if (!trafo_out.empty())
    {
      FileHandler().storeTransformations(trafo_out, trafo, {FileTypes::TRANSFORMATIONXML});
    }
    if (!in.empty()) // load input
    {
      FileTypes::Type in_type = FileHandler::getType(in);
      if (in_type == FileTypes::MZML)
      {
        PeakMap map;
        FileHandler().loadExperiment(in, map, {FileTypes::MZML}, log_type_);
        applyTransformation_( trafo, map);
        FileHandler().storeExperiment(out, map, {FileTypes::MZML}, log_type_);

      }
      else if (in_type == FileTypes::FEATUREXML)
      {
        FeatureMap map;
        FileHandler().loadFeatures(in, map, {FileTypes::FEATUREXML}, log_type_);
        applyTransformation_( trafo, map);
        FileHandler().storeFeatures(out, map, {FileTypes::FEATUREXML}, log_type_);
      }
      else if (in_type == FileTypes::CONSENSUSXML)
      {
        ConsensusMap map;
        FileHandler().loadConsensusFeatures(in, map, {FileTypes::CONSENSUSXML}, log_type_);
        applyTransformation_( trafo, map);
        FileHandler().storeConsensusFeatures(out, map, {FileTypes::CONSENSUSXML}, log_type_);
      }
      else if (in_type == FileTypes::IDXML)
      {
        vector<ProteinIdentification> proteins;
        PeptideIdentificationList peptides;
        FileHandler().loadIdentifications(in, proteins, peptides, {FileTypes::IDXML}, log_type_);
        bool store_original_rt = getFlag_("store_original_rt");
        MapAlignmentTransformer::transformRetentionTimes(peptides, trafo,
                                                         store_original_rt);
        // no "data processing" section in idXML
        FileHandler().storeIdentifications(out, proteins, peptides, {FileTypes::IDXML}, log_type_);
      }
    }

    return EXECUTION_OK;
  }

};


int main(int argc, const char** argv)
{
  TOPPMapRTTransformer tool;
  return tool.main(argc, argv);
}

/// @endcond
