// Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer: Timo Sachsenberg $
// $Authors: Nikos Patikos $
// --------------------------------------------------------------------------

#pragma once

#include <OpenMS/config.h>

namespace OpenMS
{
  class EmpiricalFormula;
  class IsotopeDistribution;

  /** 
      @brief Provides an interface for different isotope pattern generator methods.
      
      The IsotopePatternGenerator interface allows the developer integrate various 
      isotope pattern generator methods in the OpenMS code. It provides a run() method 
      that generates but does not hold any generated isotope distribution data in 
      the class. Instead it returns an IsotopeDistribution to the caller.

      Currently there are two implementations available:

      - CoarseIsotopePatternGenerator: a fast implementation of isotopic
        patterns that generates accurate probabilities for each isotopic peak
        but ignores the hyperfine isotopic structures (i.e. single peaks will
        be generated spaced ca. 1 Da apart, thus aggregating the peaks
        generated by 13C, 15N, 18O etc.).

      - FineIsotopePatternGenerator: an implementation for generation of fine
        resolution (high resolution) isotopic patterns containing the hyperfine
        isotopic structure. This will produce individual peaks for each element
        (i.e. the isotopic peaks of 13C, 15N, 18O etc. will all be individually
        resolved).

   */
  class OPENMS_DLLAPI IsotopePatternGenerator
  {
 public:
    IsotopePatternGenerator();
    IsotopePatternGenerator(double probability_cutoff);
    
    /** 
        @brief interface that is being used by the Isotope Pattern Generator methods.
        
        Method that calculates the isotope distribution for the given formula.

     */
    virtual IsotopeDistribution run(const EmpiricalFormula&) const = 0;
    virtual ~IsotopePatternGenerator();

 protected:
    double min_prob_;
  };
}

