// --------------------------------------------------------------------------
//                   OpenMS -- Open-Source Mass Spectrometry
// --------------------------------------------------------------------------
// Copyright The OpenMS Team -- Eberhard Karls University Tuebingen,
// ETH Zurich, and Freie Universitaet Berlin 2002-2022.
//
// This software is released under a three-clause BSD license:
//  * Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  * Neither the name of any author or any participating institution
//    may be used to endorse or promote products derived from this software
//    without specific prior written permission.
// For a full list of authors, refer to the file AUTHORS.
// --------------------------------------------------------------------------
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL ANY OF THE AUTHORS OR THE CONTRIBUTING
// INSTITUTIONS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
// OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
// WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
// OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
// ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// --------------------------------------------------------------------------
// $Maintainer: Julianus Pfeuffer$
// $Authors: Julianus Pfeuffer $
// --------------------------------------------------------------------------

#pragma once

#include <OpenMS/config.h>
#include <OpenMS/METADATA/PeptideIdentification.h>
#include <vector>

namespace OpenMS
{
  /**
    @brief Utilities for Feature and ConsensusMap

    @ingroup Datastructures
  */
  template <class MapType>
  class MapUtilities
  {
  public:
    /// applies a function on all PeptideHits or only assigned ones
    template <class T>
    void applyFunctionOnPeptideHits(T&& f, bool include_unassigned = true)
    {
      for (auto& feat : static_cast<MapType&>(*this))
      {
        applyFunctionOnPeptideHits_(feat.getPeptideIdentifications(), f);
      }
      if (include_unassigned)
      {
        applyFunctionOnPeptideHits_(static_cast<MapType&>(*this).getUnassignedPeptideIdentifications(), f);
      }
    }

    /// applies a function on all PeptideIDs or only assigned ones
    template <class T>
    void applyFunctionOnPeptideIDs(T&& f, bool include_unassigned = true)
    {
      for (auto& feat : static_cast<MapType&>(*this))
      {
        applyFunctionOnPeptideIDs_(feat.getPeptideIdentifications(), f);
      }
      if (include_unassigned)
      {
        applyFunctionOnPeptideIDs_(static_cast<MapType&>(*this).getUnassignedPeptideIdentifications(), f);
      }
    }

    /// applies a const function on all PeptideHits or only assigned ones
    template <class T>
    void applyFunctionOnPeptideHits(T&& f, bool include_unassigned = true) const
    {
      for (const auto& feat : static_cast<MapType const&>(*this))
      {
        applyFunctionOnPeptideHits_(feat.getPeptideIdentifications(), f);
      }
      if (include_unassigned)
      {
        applyFunctionOnPeptideHits_(static_cast<MapType const&>(*this).getUnassignedPeptideIdentifications(), f);
      }
    }

    /// applies a const function on all PeptideIDs or only assigned ones
    template <class T>
    void applyFunctionOnPeptideIDs(T&& f, bool include_unassigned = true) const
    {
      for (const auto& feat : static_cast<MapType const&>(*this))
      {
        applyFunctionOnPeptideIDs_(feat.getPeptideIdentifications(), f);
      }
      if (include_unassigned)
      {
        applyFunctionOnPeptideIDs_(static_cast<MapType const&>(*this).getUnassignedPeptideIdentifications(), f);
      }
    }

  private:
    template <class T>
    void applyFunctionOnPeptideIDs_(std::vector<PeptideIdentification>& idvec, T&& f)
    {
      for (auto& id : idvec)
      {
        f(id);
      }
    }

    template <class T>
    void applyFunctionOnPeptideHits_(std::vector<PeptideIdentification>& idvec, T&& f)
    {
      for (auto& id : idvec)
      {
        for (auto& hit : id.getHits())
        {
          f(hit);
        }
      }
    }

    template <class T>
    void applyFunctionOnPeptideIDs_(const std::vector<PeptideIdentification>& idvec, T&& f) const
    {
      for (const auto& id : idvec)
      {
        f(id);
      }
    }

    template <class T>
    void applyFunctionOnPeptideHits_(const std::vector<PeptideIdentification>& idvec, T&& f) const
    {
      for (const auto& id : idvec)
      {
        for (const auto& hit : id.getHits())
        {
          f(hit);
        }
      }
    }
  };

} // namespace OpenMS

