// --------------------------------------------------------------------------
//                   OpenMS -- Open-Source Mass Spectrometry               
// --------------------------------------------------------------------------
// Copyright The OpenMS Team -- Eberhard Karls University Tuebingen,
// ETH Zurich, and Freie Universitaet Berlin 2002-2022.
// 
// This software is released under a three-clause BSD license:
//  * Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  * Neither the name of any author or any participating institution 
//    may be used to endorse or promote products derived from this software 
//    without specific prior written permission.
// For a full list of authors, refer to the file AUTHORS. 
// --------------------------------------------------------------------------
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL ANY OF THE AUTHORS OR THE CONTRIBUTING 
// INSTITUTIONS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
// OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
// WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
// OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
// ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// --------------------------------------------------------------------------
// $Maintainer: Timo Sachsenberg $
// $Authors: $
// --------------------------------------------------------------------------

#include <OpenMS/CONCEPT/ClassTest.h>
#include <OpenMS/test_config.h>

///////////////////////////

#include <OpenMS/TRANSFORMATIONS/FEATUREFINDER/InterpolationModel.h>
#include <OpenMS/CONCEPT/Exception.h>


///////////////////////////

using namespace OpenMS;

class TestModel : public InterpolationModel
{
  public:
  TestModel()
    : InterpolationModel()
  {
    setName(getProductName());

    check_defaults_ = false;

    defaultsToParam_();
  }


  TestModel(const TestModel& source)
    : InterpolationModel(source)
  {
    updateMembers_();
  }

  ~TestModel() override
  {
  }

  virtual TestModel& operator = (const TestModel& source)
  {
    if (&source == this) return *this;

    InterpolationModel::operator = (source);
    updateMembers_();

    return *this;
  }

  void updateMembers_() override
  {
     InterpolationModel::updateMembers_();
  }

  IntensityType getIntensity(const PositionType& pos) const override
  {
    return pos[0]*3.0;
  }

  IntensityType getIntensity(CoordinateType coord) const
  {
    return coord*3.0;
  }

  bool isContained(const PositionType& pos) const override
  {
    return getIntensity(pos)>cut_off_;
  }

  void  fillIntensity(PeakType& peak) const
  {
    peak.setIntensity(getIntensity(peak.getPosition()));
  }

  void getSamples(SamplesType& /*cont*/) const override
  {
  }

  void setSamples() override
  {
  }

  CoordinateType getCenter() const override
  {
    return 10.0;
  }

  static const String getProductName()
  {
    return "TestModel";
  }

};


START_TEST(InterpolationModel , "$Id$")

/////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////

using std::stringstream;

//////////////////////////////////////

// default ctor
TestModel* ptr = nullptr;
TestModel* nullPointer = nullptr;
START_SECTION((InterpolationModel()))
	ptr = new TestModel();
	TEST_NOT_EQUAL(ptr, nullPointer)
END_SECTION

// destructor
START_SECTION((virtual ~InterpolationModel()))
	delete ptr;
END_SECTION

// assignment operator
START_SECTION((virtual InterpolationModel& operator=(const InterpolationModel &source)))
	TestModel tm1;
  TestModel tm2;

  tm1.setCutOff(3.3);
  tm2 = tm1;
	TEST_REAL_SIMILAR(tm1.getCutOff(),tm2.getCutOff())
	TEST_REAL_SIMILAR(tm1.getScalingFactor(),tm2.getScalingFactor())
END_SECTION

// copy constructor
START_SECTION((InterpolationModel(const InterpolationModel &source)))
	TestModel fp1;
  fp1.setCutOff(0.1);

  TestModel fp2(fp1);

  TestModel fp3;
  fp3.setCutOff(0.1);

  fp1 = TestModel();
	TEST_EQUAL(fp2==fp3, true)
END_SECTION

START_SECTION(([EXTRA]IntensityType getCutOff() const))
  const TestModel s;
  TEST_REAL_SIMILAR(s.getCutOff(), TestModel::IntensityType(0))
END_SECTION

START_SECTION(([EXTRA]void setCutOff(IntensityType cut_off)))
	TestModel s;
	s.setCutOff(4.4);
  TEST_REAL_SIMILAR(s.getCutOff(), 4.4)
END_SECTION

START_SECTION(([EXTRA]const String& getName() const))
	TestModel s;
  TEST_EQUAL(s.getName(), "TestModel")
END_SECTION

START_SECTION((IntensityType getIntensity(const PositionType& pos) const))
	const TestModel s;
  TestModel::PositionType pos;
  pos[0]=0.1;
  TEST_REAL_SIMILAR(s.getIntensity(pos), 0.3)
END_SECTION

START_SECTION(([EXTRA]bool isContained(const PositionType& pos) const))
	TestModel s;
  s.setCutOff(0.9);
  TestModel::PositionType pos;
  pos[0]=0.1;
  const TestModel& t = s;
  TEST_EQUAL(t.isContained(pos), false)
END_SECTION

START_SECTION(([EXTRA]void fillIntensity(PeakType& peak) const))
	const TestModel t;
  TestModel::PeakType p;
  p.getPosition()[0]=0.1;
  p.setIntensity(0.1f);
  t.fillIntensity(p);
  TEST_REAL_SIMILAR(p.getIntensity(), 0.3)
END_SECTION

START_SECTION(([EXTRA]void  fillIntensities(PeakIterator beg, PeakIterator end) const))
	const TestModel t;
  std::vector< TestModel::PeakType > vec(4);
  for (Size i=0; i<4; ++i)
  {
		vec[i].setIntensity(-0.5);
		vec[i].getPosition()[0] = i;
	}
  t.fillIntensities(vec.begin()+1, vec.end()-1);
  TEST_EQUAL(vec[0].getIntensity(), -0.5)
  TEST_EQUAL(vec[1].getIntensity(), 3.0)
  TEST_EQUAL(vec[2].getIntensity(), 6.0)
  TEST_EQUAL(vec[3].getIntensity(), -0.5)
END_SECTION

START_SECTION( virtual CoordinateType getCenter() const)
	const TestModel t;
 TEST_REAL_SIMILAR(t.getCenter(),10.0);
END_SECTION

START_SECTION([EXTRA] DefaultParmHandler::setParameters(...))
	Param p;
	p.setValue("cutoff",17.0);
	TestModel m;
	m.setParameters(p);
	TEST_REAL_SIMILAR(m.getParameters().getValue("cutoff"), 17.0)
END_SECTION

START_SECTION( void setScalingFactor(CoordinateType scaling) )
	TestModel tm;
	tm.setScalingFactor(2.0);

	TEST_REAL_SIMILAR(tm.getParameters().getValue("intensity_scaling"),2.0)
	TEST_REAL_SIMILAR(tm.getScalingFactor(),2.0)
END_SECTION

START_SECTION( void setInterpolationStep(CoordinateType interpolation_step) )
	TestModel tm;
	tm.setInterpolationStep( 10.5 );

	TEST_REAL_SIMILAR(tm.getParameters().getValue("interpolation_step"), 10.5 )
END_SECTION

START_SECTION( virtual void setSamples() )
	// not much to be tested here
END_SECTION

START_SECTION( void getSamples(SamplesType &cont) const )
	// not much to be tested here
END_SECTION

START_SECTION( virtual void setOffset(CoordinateType offset) )

END_SECTION

START_SECTION( CoordinateType getScalingFactor() const )
	TestModel tm;
	tm.setScalingFactor(666.0);

	TEST_REAL_SIMILAR(tm.getParameters().getValue("intensity_scaling"),666.0)
	TEST_REAL_SIMILAR(tm.getScalingFactor(),666.0)
END_SECTION

START_SECTION( const LinearInterpolation& getInterpolation() const )
	TestModel tm;
	InterpolationModel::LinearInterpolation interpol1;
	InterpolationModel::LinearInterpolation interpol2 = tm.getInterpolation();

	// compare models
	TEST_REAL_SIMILAR(interpol1.getScale(), interpol2.getScale());
	TEST_REAL_SIMILAR(interpol1.getInsideReferencePoint(), interpol2.getInsideReferencePoint());
	TEST_REAL_SIMILAR(interpol1.getOutsideReferencePoint(), interpol2.getOutsideReferencePoint() );

END_SECTION

START_SECTION( IntensityType getIntensity(CoordinateType coord) const )
	const TestModel s;
  TestModel::PositionType pos;
  pos[0]=0.1;
  TEST_REAL_SIMILAR(s.getIntensity(pos), 0.3)
END_SECTION

/////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////
END_TEST
