// David Eberly, Geometric Tools, Redmond WA 98052
// Copyright (c) 1998-2022
// Distributed under the Boost Software License, Version 1.0.
// https://www.boost.org/LICENSE_1_0.txt
// https://www.geometrictools.com/License/Boost/LICENSE_1_0.txt
// Version: 6.0.2022.01.06

#pragma once

#include <array>
#include <string>

// Create the lookup table for the Marching Cubes algorithm that is used to
// extract a triangular mesh that represents a level surface of a 3D image
// sampled on a regular lattice.  The assumption is that no data sample is
// zero, which allows us to have a table with 256 entries: 2 signs per
// sample, 8 samples per volume element (voxel).  Each entry corresponds to
// the pattern of 8 signs at the corners of a voxel.  The signs are stored as
// bits (b7,b6,b5,b4,b3,b2,b1,b0).  The bit assignments to voxel corners is
//   b0 = (x,y,z),   b1 = (x+1,y,z),   b2 = (x,y+1,z),   b3 = (x+1,y+1,z)
//   b4 = (x,y,z+1), b5 = (x+1,y,z+1), b6 = (x,y+1,z+1), b7 = (x+1,y+1,z+1)
// If a bit is zero, then the voxel value at the corresponding corner is
// positive; otherwise, the bit is one and the value is negative.  The
// triangles are counterclockwise ordered according to an observer viewing
// the triangle from the negative side of the level surface.

// INTERNAL USE ONLY.  This is used for convenience in creating the
// Configuration table.  TODO: Expand the macro manually for the table
// creation.
#define GTE_MC_ENTRY(name) CT_##name, &MarchingCubes::name

namespace gte
{
    class MarchingCubes
    {
    public:
        // Construction and destruction.
        virtual ~MarchingCubes() = default;

        MarchingCubes()
        {
            // Create the lookup table.
            for (mEntry = 0; mEntry < 256; ++mEntry)
            {
                (this->*ConfigurationTable()[mEntry].F)(ConfigurationTable()[mEntry].index);
            }
        }

        enum
        {
            MAX_VERTICES = 12,
            MAX_TRIANGLES = 5,
        };

        struct Topology
        {
            // All members are set to zeros.
            Topology()
                :
                numVertices(0),
                numTriangles(0)
            {
                std::fill(vpair.begin(), vpair.end(), std::array<int32_t, 2>{ 0, 0 });
                std::fill(itriple.begin(), itriple.end(), std::array<int32_t, 3>{ 0, 0, 0 });
            }

            int32_t numVertices;
            int32_t numTriangles;
            std::array<std::array<int32_t, 2>, MAX_VERTICES> vpair;
            std::array<std::array<int32_t, 3>, MAX_TRIANGLES> itriple;
        };

        // The entry must be in {0..255}.
        inline Topology const& GetTable(int32_t entry) const
        {
            return TopologyTable()[entry];
        }

        // The table has 256 entries, each 41 integers, stored as
        // table[256][41].  The return value is a pointer to the table via
        // &table[0][0].
        inline int32_t const* GetTable() const
        {
            return reinterpret_cast<int32_t const*>(TopologyTable().data());
        }

        // The pre-built topology table that is generated by the constructor.
        // This is for reference in case you want to have a GPU-based
        // implementation where you load the table as a GPU resource.
        static std::array<std::array<int32_t, 41>, 256> const& GetPrebuiltTable()
        {
            static std::array<std::array<int32_t, 41>, 256> topologyTable =
            { {
                { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 4, 0, 2, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 2, 6, 2, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 1, 3, 1, 5, 0, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 4, 2, 6, 2, 3, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 1, 0, 4, 0, 2, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 1, 5, 0, 1, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 5, 0, 4, 0, 2, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 3, 7, 1, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 6, 3, 7, 1, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 3, 7, 1, 5, 0, 1, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 4, 2, 6, 3, 7, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 5, 4, 6, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 1, 1, 3, 1, 5, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 2, 1, 3, 1, 5, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 4, 4, 5, 4, 6, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 1, 4, 5, 4, 6, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 0, 2, 2, 6, 2, 3, 1, 3, 1, 5, 0, 1, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 5, 4, 6, 2, 6, 2, 3, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 2, 2, 3, 3, 7, 1, 3, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 1, 4, 5, 4, 6, 0, 2, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 1, 2, 3, 3, 7, 1, 5, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 5, 4, 6, 0, 2, 2, 3, 3, 7, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 3, 2, 6, 3, 7, 1, 3, 0, 2, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 6, 2, 6, 3, 7, 1, 3, 0, 1, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 8, 4, 3, 7, 1, 5, 0, 1, 0, 2, 2, 6, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 5, 3, 3, 7, 2, 6, 4, 6, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 4, 0, 2, 0, 1, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 5, 7, 4, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 3, 5, 7, 4, 5, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 2, 2, 6, 2, 3, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 4, 2, 6, 2, 3, 0, 1, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 1, 3, 5, 7, 4, 5, 0, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 5, 0, 4, 2, 6, 2, 3, 1, 3, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 2, 5, 7, 4, 5, 1, 5, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 0, 1, 0, 4, 0, 2, 2, 3, 3, 7, 1, 3, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 1, 2, 3, 3, 7, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 5, 7, 4, 5, 0, 4, 0, 2, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 3, 1, 3, 0, 2, 2, 6, 3, 7, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 2, 6, 3, 7, 1, 3, 0, 1, 0, 4, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 6, 4, 5, 7, 4, 5, 0, 1, 0, 2, 2, 6, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 2, 6, 0, 4, 4, 5, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 6, 0, 4, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 4, 6, 0, 2, 0, 1, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 5, 7, 4, 6, 0, 4, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 2, 1, 3, 5, 7, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 4, 1, 5, 5, 7, 4, 6, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 1, 5, 5, 7, 4, 6, 2, 6, 2, 3, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 8, 4, 5, 7, 4, 6, 0, 4, 0, 1, 1, 3, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 5, 3, 5, 7, 1, 3, 2, 3, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 5, 7, 4, 6, 0, 4, 1, 5, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 4, 6, 0, 2, 0, 1, 1, 5, 5, 7, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 6, 4, 3, 7, 5, 7, 4, 6, 0, 4, 0, 1, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 4, 6, 5, 7, 3, 7, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 3, 7, 1, 3, 0, 2, 2, 6, 1, 5, 5, 7, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 7, 5, 3, 7, 2, 6, 4, 6, 5, 7, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 7, 5, 4, 6, 5, 7, 3, 7, 2, 6, 0, 2, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 4, 2, 2, 6, 4, 6, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 2, 0, 1, 0, 4, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 1, 3, 1, 5, 0, 1, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 2, 1, 3, 1, 5, 0, 4, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 6, 6, 7, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 3, 0, 1, 0, 4, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 0, 2, 4, 6, 6, 7, 2, 3, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 6, 6, 7, 2, 3, 1, 3, 1, 5, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 2, 3, 7, 1, 3, 2, 3, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 2, 3, 3, 7, 1, 3, 0, 1, 0, 4, 0, 2, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 3, 7, 1, 5, 0, 1, 2, 3, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 1, 5, 0, 4, 0, 2, 2, 3, 3, 7, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 5, 3, 0, 2, 4, 6, 6, 7, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 4, 6, 6, 7, 3, 7, 1, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 4, 6, 7, 3, 7, 1, 5, 0, 1, 0, 2, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 1, 5, 3, 7, 6, 7, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 6, 7, 2, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 4, 5, 6, 7, 2, 6, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 4, 5, 6, 7, 2, 6, 0, 4, 0, 1, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 2, 6, 0, 2, 1, 3, 1, 5, 4, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 6, 7, 2, 3, 0, 2, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 1, 4, 5, 6, 7, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 6, 7, 2, 3, 0, 2, 0, 4, 4, 5, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 5, 3, 6, 7, 4, 5, 1, 5, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 2, 6, 0, 4, 4, 5, 6, 7, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 4, 5, 6, 7, 2, 6, 0, 2, 0, 1, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 8, 4, 1, 5, 0, 1, 2, 3, 3, 7, 0, 4, 4, 5, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 7, 5, 6, 7, 4, 5, 1, 5, 3, 7, 2, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 3, 7, 1, 3, 0, 2, 0, 4, 4, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 4, 5, 0, 1, 1, 3, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 7, 5, 1, 5, 3, 7, 6, 7, 4, 5, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 4, 2, 4, 5, 1, 5, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 4, 5, 1, 5, 5, 7, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 4, 5, 1, 5, 5, 7, 6, 7, 2, 6, 4, 6, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 4, 5, 0, 1, 1, 3, 5, 7, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 1, 3, 5, 7, 4, 5, 0, 4, 0, 2, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 3, 6, 7, 2, 3, 0, 2, 4, 6, 4, 5, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 2, 3, 0, 1, 0, 4, 4, 6, 6, 7, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 8, 4, 4, 6, 6, 7, 2, 3, 0, 2, 5, 7, 4, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 7, 5, 5, 7, 1, 3, 2, 3, 6, 7, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 9, 3, 6, 7, 2, 6, 4, 6, 4, 5, 1, 5, 5, 7, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 12, 4, 0, 1, 0, 4, 0, 2, 2, 6, 4, 6, 6, 7, 2, 3, 3, 7, 1, 3, 1, 5, 5, 7, 4, 5, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 0, 0, 0 },
                { 8, 4, 0, 1, 2, 3, 3, 7, 5, 7, 4, 5, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 9, 5, 4, 6, 0, 4, 4, 5, 5, 7, 3, 7, 6, 7, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 8, 4, 0, 2, 4, 6, 6, 7, 3, 7, 1, 3, 4, 5, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 9, 5, 5, 7, 3, 7, 6, 7, 4, 6, 0, 4, 4, 5, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 7, 5, 4, 6, 0, 2, 0, 1, 4, 5, 5, 7, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 5, 7, 3, 7, 6, 7, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 5, 3, 0, 4, 1, 5, 5, 7, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 2, 0, 1, 1, 5, 5, 7, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 4, 6, 7, 2, 6, 0, 4, 0, 1, 1, 3, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 1, 3, 0, 2, 2, 6, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 2, 0, 4, 1, 5, 5, 7, 6, 7, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 2, 3, 6, 7, 5, 7, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 7, 5, 2, 3, 6, 7, 5, 7, 1, 3, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 4, 2, 1, 3, 2, 3, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 4, 1, 5, 5, 7, 6, 7, 2, 6, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 9, 5, 1, 3, 1, 5, 0, 1, 0, 2, 2, 6, 2, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 7, 5, 2, 3, 0, 1, 0, 4, 2, 6, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 2, 3, 0, 2, 2, 6, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 7, 5, 1, 5, 0, 4, 0, 2, 1, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 1, 5, 0, 1, 1, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 6, 2, 0, 1, 0, 4, 0, 2, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 3, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 0, 1, 0, 4, 0, 2, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 1, 5, 0, 1, 1, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 1, 5, 0, 4, 0, 2, 1, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 2, 3, 0, 2, 2, 6, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 2, 3, 0, 1, 0, 4, 2, 6, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 1, 3, 1, 5, 0, 1, 0, 2, 2, 6, 2, 3, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 4, 2, 6, 2, 3, 1, 3, 1, 5, 6, 7, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 4, 2, 1, 3, 2, 3, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 2, 3, 6, 7, 5, 7, 1, 3, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 3, 6, 7, 5, 7, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 5, 7, 1, 5, 0, 4, 0, 2, 2, 3, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 1, 3, 0, 2, 2, 6, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 6, 7, 5, 7, 1, 3, 0, 1, 0, 4, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 4, 6, 7, 5, 7, 1, 5, 0, 1, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 0, 4, 1, 5, 5, 7, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 2, 5, 7, 3, 7, 6, 7, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 4, 6, 0, 2, 0, 1, 4, 5, 5, 7, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 9, 3, 5, 7, 3, 7, 6, 7, 4, 6, 0, 4, 4, 5, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 2, 1, 3, 1, 5, 4, 5, 4, 6, 3, 7, 6, 7, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 9, 3, 4, 6, 0, 4, 4, 5, 5, 7, 3, 7, 6, 7, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 1, 4, 5, 4, 6, 2, 6, 2, 3, 5, 7, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 12, 4, 1, 3, 1, 5, 0, 1, 0, 4, 4, 5, 4, 6, 0, 2, 2, 6, 2, 3, 3, 7, 6, 7, 5, 7, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 0, 0, 0 },
                { 9, 5, 6, 7, 2, 6, 4, 6, 4, 5, 1, 5, 5, 7, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 7, 3, 5, 7, 1, 3, 2, 3, 6, 7, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 4, 5, 4, 6, 0, 2, 0, 1, 6, 7, 5, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 8, 4, 2, 3, 6, 7, 5, 7, 1, 5, 0, 1, 4, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 6, 7, 2, 3, 0, 2, 4, 6, 4, 5, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 8, 4, 1, 3, 0, 2, 2, 6, 6, 7, 5, 7, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 4, 5, 0, 1, 1, 3, 5, 7, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 9, 5, 4, 5, 1, 5, 5, 7, 6, 7, 2, 6, 4, 6, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 6, 4, 4, 5, 1, 5, 5, 7, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 4, 2, 4, 5, 1, 5, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 1, 5, 3, 7, 6, 7, 4, 5, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 4, 5, 0, 1, 1, 3, 3, 7, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 3, 7, 6, 7, 4, 5, 0, 4, 0, 2, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 3, 6, 7, 4, 5, 1, 5, 3, 7, 2, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 2, 6, 2, 3, 0, 1, 0, 4, 3, 7, 6, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 8, 4, 4, 5, 0, 1, 1, 3, 3, 7, 6, 7, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 2, 6, 0, 4, 4, 5, 6, 7, 3, 7, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 5, 3, 6, 7, 4, 5, 1, 5, 1, 3, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 6, 7, 4, 5, 1, 5, 1, 3, 2, 3, 0, 4, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 4, 2, 0, 1, 2, 3, 6, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 6, 7, 2, 3, 0, 2, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 1, 5, 1, 3, 0, 2, 2, 6, 6, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 4, 5, 6, 7, 2, 6, 0, 4, 0, 1, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 5, 3, 4, 5, 6, 7, 2, 6, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 6, 7, 2, 6, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 5, 3, 7, 6, 7, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 6, 7, 4, 6, 0, 2, 0, 1, 1, 5, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 6, 4, 3, 7, 6, 7, 4, 6, 0, 4, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 0, 2, 4, 6, 6, 7, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 1, 5, 3, 7, 6, 7, 4, 6, 0, 4, 2, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 3, 7, 1, 5, 0, 1, 2, 3, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 9, 5, 2, 3, 3, 7, 1, 3, 0, 1, 0, 4, 0, 2, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 6, 4, 3, 7, 1, 3, 2, 3, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 6, 4, 4, 6, 0, 4, 1, 5, 1, 3, 2, 3, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 0, 2, 4, 6, 6, 7, 2, 3, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 5, 3, 2, 3, 0, 1, 0, 4, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 6, 6, 7, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 5, 0, 2, 1, 3, 1, 5, 0, 4, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 2, 1, 3, 1, 5, 0, 1, 2, 6, 4, 6, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 3, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 2, 0, 1, 0, 4, 4, 6, 6, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 3, 1, 6, 7, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 2, 6, 4, 6, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 4, 6, 5, 7, 3, 7, 2, 6, 0, 2, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 7, 3, 3, 7, 2, 6, 4, 6, 5, 7, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 0, 4, 0, 2, 1, 3, 1, 5, 2, 6, 4, 6, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 4, 5, 6, 4, 6, 7, 0, 0, 0 },
                { 5, 3, 4, 6, 5, 7, 3, 7, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 3, 7, 2, 3, 0, 1, 0, 4, 4, 6, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 8, 4, 4, 6, 5, 7, 3, 7, 2, 3, 0, 2, 1, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 5, 7, 4, 6, 0, 4, 1, 5, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 5, 3, 5, 7, 1, 3, 2, 3, 2, 6, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 5, 7, 1, 3, 2, 3, 2, 6, 4, 6, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 6, 4, 1, 5, 0, 1, 2, 3, 2, 6, 4, 6, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 0, 4, 1, 5, 5, 7, 4, 6, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 4, 2, 0, 2, 4, 6, 5, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 5, 7, 4, 6, 0, 4, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 4, 6, 0, 2, 0, 1, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 4, 6, 0, 4, 1, 5, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 6, 0, 4, 4, 5, 5, 7, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 5, 7, 3, 7, 2, 6, 0, 2, 0, 1, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 8, 4, 2, 6, 0, 4, 4, 5, 5, 7, 3, 7, 0, 1, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 7, 5, 1, 3, 0, 2, 2, 6, 3, 7, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 5, 7, 3, 7, 2, 3, 0, 2, 0, 4, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 5, 3, 0, 1, 2, 3, 3, 7, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 9, 5, 0, 1, 0, 4, 0, 2, 2, 3, 3, 7, 1, 3, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 6, 4, 5, 7, 4, 5, 1, 5, 1, 3, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 6, 4, 2, 3, 2, 6, 0, 4, 4, 5, 5, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 1, 3, 5, 7, 4, 5, 0, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 7, 5, 0, 4, 2, 6, 2, 3, 0, 1, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 2, 0, 2, 2, 6, 2, 3, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 3, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 3, 5, 7, 4, 5, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 5, 7, 4, 5, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 4, 0, 2, 0, 1, 1, 5, 5, 7, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 3, 1, 5, 7, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 3, 7, 2, 6, 4, 6, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 0, 0, 0, 0, 0 },
                { 8, 4, 3, 7, 2, 6, 4, 6, 4, 5, 1, 5, 0, 2, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 5, 6, 7, 0, 0, 0 },
                { 6, 4, 1, 3, 3, 7, 2, 6, 4, 6, 4, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 2, 6, 3, 7, 1, 3, 0, 2, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 4, 2, 3, 0, 2, 4, 6, 4, 5, 1, 5, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 7, 5, 0, 1, 2, 3, 3, 7, 1, 5, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 7, 5, 0, 1, 4, 5, 4, 6, 0, 2, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 6, 5, 1, 0, 5, 2, 1, 5, 3, 2, 5, 4, 3 },
                { 6, 2, 2, 3, 3, 7, 1, 3, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 3, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 1, 3, 2, 3, 2, 6, 4, 6, 4, 5, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5, 0, 0, 0 },
                { 9, 5, 0, 2, 2, 6, 2, 3, 1, 3, 1, 5, 0, 1, 0, 4, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 1, 3, 2, 1, 4, 3, 1, 7, 4, 1, 8, 7, 0, 5, 6 },
                { 5, 3, 0, 1, 4, 5, 4, 6, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 4, 4, 5, 4, 6, 2, 6, 2, 3, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 5, 3, 0, 2, 1, 3, 1, 5, 4, 5, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 1, 1, 3, 1, 5, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 4, 2, 4, 5, 4, 6, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 4, 5, 4, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 4, 1, 5, 3, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 3, 7, 1, 5, 0, 1, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 2, 6, 3, 7, 1, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 3, 7, 1, 3, 0, 2, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 1, 5, 0, 4, 0, 2, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 1, 5, 0, 1, 2, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 0, 1, 0, 4, 0, 2, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 3, 1, 2, 3, 3, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 5, 3, 0, 4, 2, 6, 2, 3, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 0, 0, 0, 0, 0 },
                { 6, 4, 1, 3, 1, 5, 0, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 5, 1, 5, 4, 1, 4, 3, 1, 3, 2, 0, 0, 0 },
                { 4, 2, 2, 6, 2, 3, 0, 1, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 0, 2, 2, 6, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 4, 2, 0, 4, 0, 2, 1, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 1, 3, 1, 5, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 3, 1, 0, 1, 0, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
                { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
            }};
            return topologyTable;
        }

        // Get the configuration type for the voxel, which is one of the
        // string names of the 'void Bits* (int32_t[8])' functions.
        static std::string GetConfigurationType(int32_t entry)
        {
            if (0 <= entry && entry < 256)
            {
                return ConfigurationString()[ConfigurationTable()[entry].type];
            }
            return "";
        }

    protected:
        // Support for lookup construction and access.
        // mTable[i][0] = numVertices
        // mTable[i][1] = numTriangles
        // mTable[i][2..25] = pairs of corner indices (maximum of 12 pairs)
        // mTable[i][26..40] = triples of indices (maximum of 5 triples)
        static std::array<Topology, 256>& TopologyTable()
        {
            static std::array<Topology, 256> topologyTable;
            return topologyTable;
        }

        // The constructor iterates mEntry from 0 to 255 and calls configuration
        // functions, each calling SetTable(...).  The mEntry value is the table
        // index to be used.
        int32_t mEntry;

        void SetTable(int32_t numV, int32_t const* vpair, int32_t numT, int32_t const* itriple)
        {
            // The item is already zeroed in the constructor.
            Topology& topology = TopologyTable()[mEntry];
            topology.numVertices = numV;
            topology.numTriangles = numT;

            // Store vertex pairs with minimum index occurring first.
            for (int32_t i = 0; i < numV; ++i, vpair += 2)
            {
                topology.vpair[i][0] = std::min(vpair[0], vpair[1]);
                topology.vpair[i][1] = std::max(vpair[0], vpair[1]);
            }

            // Store triangle triples as is.
            for (int32_t i = 0; i < numT; ++i, itriple += 3)
            {
                topology.itriple[i] = { itriple[0], itriple[1], itriple[2] };
            }
        }

        // The precomputed information about the 256 configurations for voxels.
        void Bits0(int32_t[8])
        {
            SetTable(0, nullptr, 0, nullptr);
        }

        void Bits1(int32_t index[8])
        {
            int32_t const numV = 3;
            int32_t vpair[2 * numV] =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0]
            };

            int32_t const numT = 1;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits7(int32_t index[8])
        {
            int32_t const numV = 3;
            int32_t vpair[2 * numV] =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0]
            };

            int32_t const numT = 1;
            int32_t itriple[3 * numT] =
            {
                0, 2, 1
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits2Edge(int32_t index[8])
        {
            int32_t const numV = 4;
            int32_t vpair[2 * numV] =
            {
                index[4], index[0],
                index[2], index[0],
                index[3], index[1],
                index[5], index[1]
            };

            int32_t const numT = 2;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits6Edge(int32_t index[8])
        {
            int32_t const numV = 4;
            int32_t vpair[2 * numV] =
            {
                index[4], index[0],
                index[2], index[0],
                index[3], index[1],
                index[5], index[1]
            };

            int32_t const numT = 2;
            int32_t itriple[3 * numT] =
            {
                0, 2, 1,
                0, 3, 2
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits2FaceDiag(int32_t index[8])
        {
            int32_t const numV = 6;
            int32_t vpair[2 * numV] =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0],
                index[2], index[3],
                index[7], index[3],
                index[1], index[3]
            };

            int32_t const numT = 2;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                3, 4, 5
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits6FaceDiag(int32_t index[8])
        {
            int32_t const numV = 6;
            int32_t vpair[2 * numV] =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0],
                index[2], index[3],
                index[7], index[3],
                index[1], index[3]
            };

            // Not the reverse ordering from Bit2FaceDiag due to ambiguous face
            // handling.
            int32_t const numT = 4;
            int32_t itriple[3 * numT] =
            {
                1, 0, 5,
                1, 5, 4,
                1, 4, 3,
                1, 3, 2
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits2BoxDiag(int32_t index[8])
        {
            int32_t const numV = 6;
            int32_t vpair[2 * numV] =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0],
                index[3], index[7],
                index[6], index[7],
                index[5], index[7]
            };

            int32_t const numT = 2;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                3, 4, 5
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits6BoxDiag(int32_t index[8])
        {
            int32_t const numV = 6;
            int32_t vpair[2 * numV] =
            {
                index[1], index[0],
                index[4], index[0],
                index[2], index[0],
                index[3], index[7],
                index[6], index[7],
                index[5], index[7]
            };

            int32_t const numT = 2;
            int32_t itriple[3 * numT] =
            {
                0, 2, 1,
                3, 5, 4
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits3SameFace(int32_t index[8])
        {
            int32_t const numV = 5;
            int32_t vpair[2 * numV] =
            {
                index[4], index[0],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5]
            };

            int32_t const numT = 3;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits5SameFace(int32_t index[8])
        {
            int32_t const numV = 5;
            int32_t vpair[2 * numV] =
            {
                index[4], index[0],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5]
            };

            int32_t const numT = 3;
            int32_t itriple[3 * numT] =
            {
                0, 2, 1,
                0, 3, 2,
                0, 4, 3
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits3EdgeFaceDiag(int32_t index[8])
        {
            int32_t const numV = 7;
            int32_t vpair[2 * numV] =
            {
                index[0], index[1],
                index[4], index[5],
                index[4], index[6],
                index[0], index[2],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3]
            };

            int32_t const numT = 3;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3,
                4, 5, 6
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits5EdgeFaceDiag(int32_t index[8])
        {
            int32_t const numV = 7;
            int32_t vpair[2 * numV] =
            {
                index[0], index[1],
                index[4], index[5],
                index[4], index[6],
                index[0], index[2],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3]
            };

            // Not the reverse ordering from Bit3EdgeFaceDiag due to ambiguous face
            // handling.
            int32_t const numT = 5;
            int32_t itriple[3 * numT] =
            {
                5, 0, 6,
                5, 1, 0,
                5, 2, 1,
                5, 3, 2,
                5, 4, 3
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits3FaceDiagFaceDiag(int32_t index[8])
        {
            int32_t const numV = 9;
            int32_t vpair[2 * numV] =
            {
                index[0], index[1],
                index[0], index[4],
                index[0], index[2],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3],
                index[1], index[5],
                index[5], index[7],
                index[4], index[5]
            };

            int32_t const numT = 3;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                3, 4, 5,
                6, 7, 8
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits5FaceDiagFaceDiag(int32_t index[8])
        {
            int32_t const numV = 9;
            int32_t vpair[2 * numV] =
            {
                index[0], index[1],
                index[0], index[4],
                index[0], index[2],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3],
                index[1], index[5],
                index[5], index[7],
                index[4], index[5]
            };

            // Not the reverse ordering from Bit3FaceDiagFaceDiag due to ambiguous face
            // handling.
            int32_t const numT = 5;
            int32_t itriple[3 * numT] =
            {
                1, 3, 2,
                1, 4, 3,
                1, 7, 4,
                1, 8, 7,
                0, 5, 6
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits4SameFace(int32_t index[8])
        {
            int32_t const numV = 4;
            int32_t vpair[2 * numV] =
            {
                index[0], index[4],
                index[2], index[6],
                index[3], index[7],
                index[1], index[5]
            };

            int32_t const numT = 2;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits4FaceEdge(int32_t index[8])
        {
            int32_t const numV = 6;
            int32_t vpair[2 * numV] =
            {
                index[4], index[5],
                index[4], index[6],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5]
            };

            int32_t const numT = 4;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4,
                0, 4, 5
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits4FaceFaceDiagL(int32_t index[8])
        {
            int32_t const numV = 6;
            int32_t vpair[2 * numV] =
            {
                index[4], index[5],
                index[0], index[4],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[5], index[7]
            };

            int32_t const numT = 4;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4,
                0, 4, 5
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits4FaceFaceDiagR(int32_t index[8])
        {
            int32_t const numV = 6;
            int32_t vpair[2 * numV] =
            {
                index[4], index[6],
                index[6], index[7],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5],
                index[0], index[4]
            };

            int32_t const numT = 4;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4,
                0, 4, 5
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits4FaceBoxDiag(int32_t index[8])
        {
            int32_t const numV = 8;
            int32_t vpair[2 * numV] =
            {
                index[0], index[4],
                index[2], index[6],
                index[2], index[3],
                index[1], index[3],
                index[1], index[5],
                index[6], index[7],
                index[5], index[7],
                index[3], index[7]
            };

            int32_t const numT = 4;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3,
                0, 3, 4,
                5, 6, 7
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits4EdgeEdgePara(int32_t index[8])
        {
            int32_t const numV = 8;
            int32_t vpair[2 * numV] =
            {
                index[0], index[4],
                index[0], index[2],
                index[1], index[3],
                index[1], index[5],
                index[2], index[6],
                index[4], index[6],
                index[5], index[7],
                index[3], index[7]
            };

            int32_t const numT = 4;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                0, 2, 3,
                4, 5, 6,
                4, 6, 7
            };

            SetTable(numV, vpair, numT, itriple);
        }

        void Bits4EdgeEdgePerp(int32_t index[8])
        {
            int32_t const numV = 12;
            int32_t vpair[2 * numV] =
            {
                index[0], index[1],
                index[0], index[4],
                index[0], index[2],
                index[2], index[6],
                index[4], index[6],
                index[6], index[7],
                index[2], index[3],
                index[3], index[7],
                index[1], index[3],
                index[1], index[5],
                index[5], index[7],
                index[4], index[5]
            };

            int32_t const numT = 4;
            int32_t itriple[3 * numT] =
            {
                0, 1, 2,
                3, 4, 5,
                6, 7, 8,
                9, 10, 11
            };

            SetTable(numV, vpair, numT, itriple);
        }

        enum ConfigurationType
        {
            CT_Bits0,
            CT_Bits1,
            CT_Bits7,
            CT_Bits2Edge,
            CT_Bits6Edge,
            CT_Bits2FaceDiag,
            CT_Bits6FaceDiag,
            CT_Bits2BoxDiag,
            CT_Bits6BoxDiag,
            CT_Bits3SameFace,
            CT_Bits5SameFace,
            CT_Bits3EdgeFaceDiag,
            CT_Bits5EdgeFaceDiag,
            CT_Bits3FaceDiagFaceDiag,
            CT_Bits5FaceDiagFaceDiag,
            CT_Bits4SameFace,
            CT_Bits4FaceEdge,
            CT_Bits4FaceFaceDiagL,
            CT_Bits4FaceFaceDiagR,
            CT_Bits4FaceBoxDiag,
            CT_Bits4EdgeEdgePara,
            CT_Bits4EdgeEdgePerp,
            CT_NUM_TYPES
        };

        typedef void (MarchingCubes::*Function)(int32_t[8]);

        struct Configuration
        {
            ConfigurationType type;
            Function F;
            int32_t index[8];
        };

        static std::array<Configuration, 256>& ConfigurationTable()
        {
            static std::array<Configuration, 256> configuration =
            {{
                /*00000000*/{ GTE_MC_ENTRY(Bits0), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00000001*/{ GTE_MC_ENTRY(Bits1), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00000010*/{ GTE_MC_ENTRY(Bits1), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*00000011*/{ GTE_MC_ENTRY(Bits2Edge), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00000100*/{ GTE_MC_ENTRY(Bits1), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*00000101*/{ GTE_MC_ENTRY(Bits2Edge), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*00000110*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*00000111*/{ GTE_MC_ENTRY(Bits3SameFace), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00001000*/{ GTE_MC_ENTRY(Bits1), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*00001001*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00001010*/{ GTE_MC_ENTRY(Bits2Edge), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*00001011*/{ GTE_MC_ENTRY(Bits3SameFace), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*00001100*/{ GTE_MC_ENTRY(Bits2Edge), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*00001101*/{ GTE_MC_ENTRY(Bits3SameFace), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*00001110*/{ GTE_MC_ENTRY(Bits3SameFace), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*00001111*/{ GTE_MC_ENTRY(Bits4SameFace), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00010000*/{ GTE_MC_ENTRY(Bits1), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*00010001*/{ GTE_MC_ENTRY(Bits2Edge), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*00010010*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 1, 0, 5, 4, 3, 2, 7, 6 }},
                /*00010011*/{ GTE_MC_ENTRY(Bits3SameFace), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*00010100*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*00010101*/{ GTE_MC_ENTRY(Bits3SameFace), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*00010110*/{ GTE_MC_ENTRY(Bits3FaceDiagFaceDiag), {  2, 0, 3, 1, 6, 4, 7, 5 }},
                /*00010111*/{ GTE_MC_ENTRY(Bits4FaceEdge), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00011000*/{ GTE_MC_ENTRY(Bits2BoxDiag), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*00011001*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00011010*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 1, 0, 5, 4, 3, 2, 7, 6 }},
                /*00011011*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*00011100*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 2, 6, 0, 4, 3, 7, 1, 5 }},
                /*00011101*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*00011110*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*00011111*/{ GTE_MC_ENTRY(Bits5SameFace), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*00100000*/{ GTE_MC_ENTRY(Bits1), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*00100001*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*00100010*/{ GTE_MC_ENTRY(Bits2Edge), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*00100011*/{ GTE_MC_ENTRY(Bits3SameFace), { 1, 0, 5, 4, 3, 2, 7, 6 }},
                /*00100100*/{ GTE_MC_ENTRY(Bits2BoxDiag), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*00100101*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*00100110*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*00100111*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00101000*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*00101001*/{ GTE_MC_ENTRY(Bits3FaceDiagFaceDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*00101010*/{ GTE_MC_ENTRY(Bits3SameFace), { 1, 5, 3, 7, 0, 4, 2, 6 }},
                /*00101011*/{ GTE_MC_ENTRY(Bits4FaceEdge), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*00101100*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 3, 1, 7, 5, 2, 0, 6, 4 }},
                /*00101101*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*00101110*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*00101111*/{ GTE_MC_ENTRY(Bits5SameFace), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*00110000*/{ GTE_MC_ENTRY(Bits2Edge), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*00110001*/{ GTE_MC_ENTRY(Bits3SameFace), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*00110010*/{ GTE_MC_ENTRY(Bits3SameFace), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*00110011*/{ GTE_MC_ENTRY(Bits4SameFace), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*00110100*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*00110101*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*00110110*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*00110111*/{ GTE_MC_ENTRY(Bits5SameFace), { 7, 6, 3, 2, 5, 4, 1, 0 }},
                /*00111000*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*00111001*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*00111010*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*00111011*/{ GTE_MC_ENTRY(Bits5SameFace), { 6, 2, 7, 3, 4, 0, 5, 1 }},
                /*00111100*/{ GTE_MC_ENTRY(Bits4EdgeEdgePara), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*00111101*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 7, 3, 5, 1, 6, 2, 4, 0 }},
                /*00111110*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*00111111*/{ GTE_MC_ENTRY(Bits6Edge), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*01000000*/{ GTE_MC_ENTRY(Bits1), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*01000001*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*01000010*/{ GTE_MC_ENTRY(Bits2BoxDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*01000011*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*01000100*/{ GTE_MC_ENTRY(Bits2Edge), { 6, 2, 7, 3, 4, 0, 5, 1 }},
                /*01000101*/{ GTE_MC_ENTRY(Bits3SameFace), { 2, 6, 0, 4, 3, 7, 1, 5 }},
                /*01000110*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*01000111*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*01001000*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*01001001*/{ GTE_MC_ENTRY(Bits3FaceDiagFaceDiag), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*01001010*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*01001011*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*01001100*/{ GTE_MC_ENTRY(Bits3SameFace), { 2, 3, 6, 7, 0, 1, 4, 5 }},
                /*01001101*/{ GTE_MC_ENTRY(Bits4FaceEdge), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*01001110*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*01001111*/{ GTE_MC_ENTRY(Bits5SameFace), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*01010000*/{ GTE_MC_ENTRY(Bits2Edge), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*01010001*/{ GTE_MC_ENTRY(Bits3SameFace), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*01010010*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*01010011*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*01010100*/{ GTE_MC_ENTRY(Bits3SameFace), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*01010101*/{ GTE_MC_ENTRY(Bits4SameFace), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*01010110*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*01010111*/{ GTE_MC_ENTRY(Bits5SameFace), { 7, 3, 5, 1, 6, 2, 4, 0 }},
                /*01011000*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 6, 2, 7, 3, 4, 0, 5, 1 }},
                /*01011001*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*01011010*/{ GTE_MC_ENTRY(Bits4EdgeEdgePara), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*01011011*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 7, 6, 3, 2, 5, 4, 1, 0 }},
                /*01011100*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*01011101*/{ GTE_MC_ENTRY(Bits5SameFace), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*01011110*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*01011111*/{ GTE_MC_ENTRY(Bits6Edge), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*01100000*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*01100001*/{ GTE_MC_ENTRY(Bits3FaceDiagFaceDiag), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*01100010*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*01100011*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 1, 0, 5, 4, 3, 2, 7, 6 }},
                /*01100100*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*01100101*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 2, 6, 0, 4, 3, 7, 1, 5 }},
                /*01100110*/{ GTE_MC_ENTRY(Bits4EdgeEdgePara), { 6, 2, 7, 3, 4, 0, 5, 1 }},
                /*01100111*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*01101000*/{ GTE_MC_ENTRY(Bits3FaceDiagFaceDiag), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*01101001*/{ GTE_MC_ENTRY(Bits4EdgeEdgePerp), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*01101010*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 1, 5, 3, 7, 0, 4, 2, 6 }},
                /*01101011*/{ GTE_MC_ENTRY(Bits5FaceDiagFaceDiag), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*01101100*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 2, 3, 6, 7, 0, 1, 4, 5 }},
                /*01101101*/{ GTE_MC_ENTRY(Bits5FaceDiagFaceDiag), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*01101110*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*01101111*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*01110000*/{ GTE_MC_ENTRY(Bits3SameFace), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*01110001*/{ GTE_MC_ENTRY(Bits4FaceEdge), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*01110010*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*01110011*/{ GTE_MC_ENTRY(Bits5SameFace), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*01110100*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*01110101*/{ GTE_MC_ENTRY(Bits5SameFace), { 3, 1, 7, 5, 2, 0, 6, 4 }},
                /*01110110*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*01110111*/{ GTE_MC_ENTRY(Bits6Edge), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*01111000*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*01111001*/{ GTE_MC_ENTRY(Bits5FaceDiagFaceDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*01111010*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 2, 3, 6, 7, 0, 1, 4, 5 }},
                /*01111011*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 2, 3, 6, 7, 0, 1, 4, 5 }},
                /*01111100*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 1, 5, 3, 7, 0, 4, 2, 6 }},
                /*01111101*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 1, 5, 3, 7, 0, 4, 2, 6 }},
                /*01111110*/{ GTE_MC_ENTRY(Bits6BoxDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*01111111*/{ GTE_MC_ENTRY(Bits7), { 7, 3, 5, 1, 6, 2, 4, 0 }},
                /*10000000*/{ GTE_MC_ENTRY(Bits1), { 7, 3, 5, 1, 6, 2, 4, 0 }},
                /*10000001*/{ GTE_MC_ENTRY(Bits2BoxDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*10000010*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 1, 5, 3, 7, 0, 4, 2, 6 }},
                /*10000011*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 1, 5, 3, 7, 0, 4, 2, 6 }},
                /*10000100*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 2, 3, 6, 7, 0, 1, 4, 5 }},
                /*10000101*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 2, 3, 6, 7, 0, 1, 4, 5 }},
                /*10000110*/{ GTE_MC_ENTRY(Bits3FaceDiagFaceDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*10000111*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*10001000*/{ GTE_MC_ENTRY(Bits2Edge), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*10001001*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*10001010*/{ GTE_MC_ENTRY(Bits3SameFace), { 3, 1, 7, 5, 2, 0, 6, 4 }},
                /*10001011*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*10001100*/{ GTE_MC_ENTRY(Bits3SameFace), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*10001101*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*10001110*/{ GTE_MC_ENTRY(Bits4FaceEdge), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*10001111*/{ GTE_MC_ENTRY(Bits5SameFace), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*10010000*/{ GTE_MC_ENTRY(Bits2FaceDiag), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*10010001*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*10010010*/{ GTE_MC_ENTRY(Bits3FaceDiagFaceDiag), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*10010011*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*10010100*/{ GTE_MC_ENTRY(Bits3FaceDiagFaceDiag), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*10010101*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*10010110*/{ GTE_MC_ENTRY(Bits4EdgeEdgePerp), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*10010111*/{ GTE_MC_ENTRY(Bits5FaceDiagFaceDiag), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*10011000*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*10011001*/{ GTE_MC_ENTRY(Bits4EdgeEdgePara), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*10011010*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 3, 1, 7, 5, 2, 0, 6, 4 }},
                /*10011011*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*10011100*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*10011101*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*10011110*/{ GTE_MC_ENTRY(Bits5FaceDiagFaceDiag), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*10011111*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*10100000*/{ GTE_MC_ENTRY(Bits2Edge), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*10100001*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*10100010*/{ GTE_MC_ENTRY(Bits3SameFace), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*10100011*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 1, 0, 5, 4, 3, 2, 7, 6 }},
                /*10100100*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 7, 6, 3, 2, 5, 4, 1, 0 }},
                /*10100101*/{ GTE_MC_ENTRY(Bits4EdgeEdgePara), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*10100110*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*10100111*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 6, 2, 7, 3, 4, 0, 5, 1 }},
                /*10101000*/{ GTE_MC_ENTRY(Bits3SameFace), { 7, 3, 5, 1, 6, 2, 4, 0 }},
                /*10101001*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 7, 3, 5, 1, 6, 2, 4, 0 }},
                /*10101010*/{ GTE_MC_ENTRY(Bits4SameFace), { 1, 5, 3, 7, 0, 4, 2, 6 }},
                /*10101011*/{ GTE_MC_ENTRY(Bits5SameFace), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*10101100*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*10101101*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*10101110*/{ GTE_MC_ENTRY(Bits5SameFace), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*10101111*/{ GTE_MC_ENTRY(Bits6Edge), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*10110000*/{ GTE_MC_ENTRY(Bits3SameFace), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*10110001*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*10110010*/{ GTE_MC_ENTRY(Bits4FaceEdge), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*10110011*/{ GTE_MC_ENTRY(Bits5SameFace), { 2, 3, 6, 7, 0, 1, 4, 5 }},
                /*10110100*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 5, 4, 7, 6, 1, 0, 3, 2 }},
                /*10110101*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*10110110*/{ GTE_MC_ENTRY(Bits5FaceDiagFaceDiag), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*10110111*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 3, 7, 2, 6, 1, 5, 0, 4 }},
                /*10111000*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 7, 3, 5, 1, 6, 2, 4, 0 }},
                /*10111001*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*10111010*/{ GTE_MC_ENTRY(Bits5SameFace), { 2, 6, 0, 4, 3, 7, 1, 5 }},
                /*10111011*/{ GTE_MC_ENTRY(Bits6Edge), { 6, 2, 7, 3, 4, 0, 5, 1 }},
                /*10111100*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*10111101*/{ GTE_MC_ENTRY(Bits6BoxDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*10111110*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*10111111*/{ GTE_MC_ENTRY(Bits7), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*11000000*/{ GTE_MC_ENTRY(Bits2Edge), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*11000001*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*11000010*/{ GTE_MC_ENTRY(Bits3EdgeFaceDiag), { 7, 3, 5, 1, 6, 2, 4, 0 }},
                /*11000011*/{ GTE_MC_ENTRY(Bits4EdgeEdgePara), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*11000100*/{ GTE_MC_ENTRY(Bits3SameFace), { 6, 2, 7, 3, 4, 0, 5, 1 }},
                /*11000101*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 2, 6, 0, 4, 3, 7, 1, 5 }},
                /*11000110*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 6, 2, 7, 3, 4, 0, 5, 1 }},
                /*11000111*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*11001000*/{ GTE_MC_ENTRY(Bits3SameFace), { 7, 6, 3, 2, 5, 4, 1, 0 }},
                /*11001001*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 7, 6, 3, 2, 5, 4, 1, 0 }},
                /*11001010*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 7, 6, 3, 2, 5, 4, 1, 0 }},
                /*11001011*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*11001100*/{ GTE_MC_ENTRY(Bits4SameFace), { 2, 3, 6, 7, 0, 1, 4, 5 }},
                /*11001101*/{ GTE_MC_ENTRY(Bits5SameFace), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*11001110*/{ GTE_MC_ENTRY(Bits5SameFace), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*11001111*/{ GTE_MC_ENTRY(Bits6Edge), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*11010000*/{ GTE_MC_ENTRY(Bits3SameFace), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*11010001*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*11010010*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*11010011*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 3, 1, 7, 5, 2, 0, 6, 4 }},
                /*11010100*/{ GTE_MC_ENTRY(Bits4FaceEdge), { 6, 4, 2, 0, 7, 5, 3, 1 }},
                /*11010101*/{ GTE_MC_ENTRY(Bits5SameFace), { 1, 5, 3, 7, 0, 4, 2, 6 }},
                /*11010110*/{ GTE_MC_ENTRY(Bits5FaceDiagFaceDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*11010111*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*11011000*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 6, 7, 4, 5, 2, 3, 0, 1 }},
                /*11011001*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*11011010*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*11011011*/{ GTE_MC_ENTRY(Bits6BoxDiag), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*11011100*/{ GTE_MC_ENTRY(Bits5SameFace), { 1, 0, 5, 4, 3, 2, 7, 6 }},
                /*11011101*/{ GTE_MC_ENTRY(Bits6Edge), { 5, 1, 4, 0, 7, 3, 6, 2 }},
                /*11011110*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*11011111*/{ GTE_MC_ENTRY(Bits7), { 5, 7, 1, 3, 4, 6, 0, 2 }},
                /*11100000*/{ GTE_MC_ENTRY(Bits3SameFace), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*11100001*/{ GTE_MC_ENTRY(Bits4FaceBoxDiag), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*11100010*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagL), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*11100011*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 2, 6, 0, 4, 3, 7, 1, 5 }},
                /*11100100*/{ GTE_MC_ENTRY(Bits4FaceFaceDiagR), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*11100101*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 1, 0, 5, 4, 3, 2, 7, 6 }},
                /*11100110*/{ GTE_MC_ENTRY(Bits5EdgeFaceDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*11100111*/{ GTE_MC_ENTRY(Bits6BoxDiag), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*11101000*/{ GTE_MC_ENTRY(Bits4FaceEdge), { 7, 5, 6, 4, 3, 1, 2, 0 }},
                /*11101001*/{ GTE_MC_ENTRY(Bits5FaceDiagFaceDiag), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*11101010*/{ GTE_MC_ENTRY(Bits5SameFace), { 0, 2, 4, 6, 1, 3, 5, 7 }},
                /*11101011*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*11101100*/{ GTE_MC_ENTRY(Bits5SameFace), { 0, 4, 1, 5, 2, 6, 3, 7 }},
                /*11101101*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 1, 0, 5, 4, 3, 2, 7, 6 }},
                /*11101110*/{ GTE_MC_ENTRY(Bits6Edge), { 4, 0, 6, 2, 5, 1, 7, 3 }},
                /*11101111*/{ GTE_MC_ENTRY(Bits7), { 4, 5, 0, 1, 6, 7, 2, 3 }},
                /*11110000*/{ GTE_MC_ENTRY(Bits4SameFace), { 4, 6, 5, 7, 0, 2, 1, 3 }},
                /*11110001*/{ GTE_MC_ENTRY(Bits5SameFace), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*11110010*/{ GTE_MC_ENTRY(Bits5SameFace), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*11110011*/{ GTE_MC_ENTRY(Bits6Edge), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*11110100*/{ GTE_MC_ENTRY(Bits5SameFace), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*11110101*/{ GTE_MC_ENTRY(Bits6Edge), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*11110110*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*11110111*/{ GTE_MC_ENTRY(Bits7), { 3, 2, 1, 0, 7, 6, 5, 4 }},
                /*11111000*/{ GTE_MC_ENTRY(Bits5SameFace), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*11111001*/{ GTE_MC_ENTRY(Bits6FaceDiag), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*11111010*/{ GTE_MC_ENTRY(Bits6Edge), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*11111011*/{ GTE_MC_ENTRY(Bits7), { 2, 0, 3, 1, 6, 4, 7, 5 }},
                /*11111100*/{ GTE_MC_ENTRY(Bits6Edge), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*11111101*/{ GTE_MC_ENTRY(Bits7), { 1, 3, 0, 2, 5, 7, 4, 6 }},
                /*11111110*/{ GTE_MC_ENTRY(Bits7), { 0, 1, 2, 3, 4, 5, 6, 7 }},
                /*11111111*/{ GTE_MC_ENTRY(Bits0), { 0, 1, 2, 3, 4, 5, 6, 7 }}
            }};
            return configuration;
        }

        static std::array<std::string, CT_NUM_TYPES>& ConfigurationString()
        {
            static std::array<std::string, CT_NUM_TYPES> configurationString =
            {
                "Bits0",
                "Bits1",
                "Bits7",
                "Bits2Edge",
                "Bits6Edge",
                "Bits2FaceDiag",
                "Bits6FaceDiag",
                "Bits2BoxDiag",
                "Bits6BoxDiag",
                "Bits3SameFace",
                "Bits5SameFace",
                "Bits3EdgeFaceDiag",
                "Bits5EdgeFaceDiag",
                "Bits3FaceDiagFaceDiag",
                "Bits5FaceDiagFaceDiag",
                "Bits4SameFace",
                "Bits4FaceEdge",
                "Bits4FaceFaceDiagL",
                "Bits4FaceFaceDiagR",
                "Bits4FaceBoxDiag",
                "Bits4EdgeEdgePara",
                "Bits4EdgeEdgePerp"
            };
            return configurationString;
        }
    };
}
