# Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
# SPDX-License-Identifier: BSD-3-Clause
#
# --------------------------------------------------------------------------
# $Maintainer: Julianus Pfeuffer $
# $Authors: Stephan Aiche, Chris Bielow, Julianus Pfeuffer $
# --------------------------------------------------------------------------

# required modules
include(CMakeParseArguments)
include(GenerateExportHeader)
include(CheckLibArchitecture)

#------------------------------------------------------------------------------
## export a single option indicating if libraries should be build as unity
## build
option(ENABLE_UNITYBUILD "Enables unity builds for all libraries." OFF)

#------------------------------------------------------------------------------
## Unity Build of a set of cpp files
## i.e., make one large compilation unit which usually compiles a lot faster
##
## pros:
##    - compiles a lot faster
##    - reveals double definitions of internal classes in different cpp files
##    - reveals weird variable names (like 'IN'), which are defined as macro in certain headers which are now part of the compilation unit
## cons:
##    - not desirable for developing, since a small change will trigger recreation of the whole lib
function(convert_to_unity_build UB_SUFFIX SOURCE_FILES_NAME)
   set(files ${${SOURCE_FILES_NAME}})
   # generate a unique file name for the unity build translation unit
   set(unit_build_file ${CMAKE_CURRENT_BINARY_DIR}/ub_${UB_SUFFIX}.cpp)
   # exclude all translation units from compilation
   set_source_files_properties(${files} PROPERTIES HEADER_FILE_ONLY true)
   # open the unity build file
   file(WRITE ${unit_build_file} "// Unity Build generated by CMake\n")
   # add include statement for each translation unit
   foreach(source_file ${files})
     # we have headers in there as well, which should not be included explicitly
     if (${source_file} MATCHES "\\.cpp|\\.cxx") # cxx for moc's;
       if (IS_ABSOLUTE ${source_file})
         file(APPEND ${unit_build_file} "#include<${source_file}>\n")
       else()
         file(APPEND ${unit_build_file} "#include<${CMAKE_CURRENT_SOURCE_DIR}/${source_file}>\n")
       endif()
     endif()
   endforeach(source_file)
   # add unity build aggregate as source file
   set(${SOURCE_FILES_NAME} ${${SOURCE_FILES_NAME}} ${unit_build_file} PARENT_SCOPE)
endfunction(convert_to_unity_build)

#------------------------------------------------------------------------------
## Copy the dll produced by the given target to the test/doc binary path.
## @param targetname The target to modify.
## @note This macro will do nothing outside of Windows since the linker will find the libs.
macro(copy_dll_to_extern_bin targetname)
  if (WIN32)
    if (CMAKE_GENERATOR MATCHES "Visual Studio")
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/src/tests/class_tests/bin/$(ConfigurationName)/$(TargetFileName)" DLL_TEST_TARGET)
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/src/tests/class_tests/bin/$(ConfigurationName)" DLL_TEST_TARGET_PATH)

      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/doc/doxygen/parameters/$(ConfigurationName)/$(TargetFileName)" DLL_DOC_TARGET)
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/doc/doxygen/parameters/$(ConfigurationName)" DLL_DOC_TARGET_PATH)
    elseif(NOT GENERATOR_IS_MULTI_CONFIG)
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/src/tests/class_tests/bin/" DLL_TEST_TARGET)
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/src/tests/class_tests/bin/" DLL_TEST_TARGET_PATH)

      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/doc/doxygen/parameters/" DLL_DOC_TARGET)
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/doc/doxygen/parameters/" DLL_DOC_TARGET_PATH)
    else()
      message(WARNING "Sorry, multiconfig generators on windows other than Visual Studio not supported yet.
              Please look for the line of this error and implement some CMake Generator expressions to copy
              DLLs to the binaries, or modify your environment for the tests to find all library DLLs.")
    endif()
    add_custom_command(TARGET ${targetname}
            POST_BUILD
            COMMAND ${CMAKE_COMMAND} -E make_directory "${DLL_TEST_TARGET_PATH}"
            COMMAND ${CMAKE_COMMAND} -E copy_if_different $<TARGET_FILE:${targetname}> ${DLL_TEST_TARGET}
            COMMAND ${CMAKE_COMMAND} -E make_directory "${DLL_DOC_TARGET_PATH}"
            COMMAND ${CMAKE_COMMAND} -E copy_if_different $<TARGET_FILE:${targetname}> ${DLL_DOC_TARGET}
            )
  endif()
endmacro()

#------------------------------------------------------------------------------
# openms_add_library()
# Create an OpenMS library, install it, register for export of targets, and
# export all required variables for later usage in the build system.
#
# Signature:
# openms_add_library(TARGET_NAME  OpenMS
#                    SOURCE_FILES  <source files to build the library>
#                    HEADER_FILES  <header files associated to the library>
#                                  (will be installed with the library)
#                    INTERNAL_INCLUDES <list of internal include directories for the library>
#                    PRIVATE_INCLUDES <list of include directories that will be used for compilation but that will not be exposed to other libraries>
#                    EXTERNAL_INCLUDES <list of external include directories for the library>
#                                      (will be added with -isystem if available)
#                    LINK_LIBRARIES <list of libraries used when linking the library>
#                    PRIVATE_LINK_LIBRARIES <list of internal libraries used when linking the library>
#                    DLL_EXPORT_PATH <path to the dll export header>)
function(openms_add_library)
  #------------------------------------------------------------------------------
  # parse arguments to function
  set(options )
  set(oneValueArgs TARGET_NAME DLL_EXPORT_PATH)
  set(multiValueArgs INTERNAL_INCLUDES PRIVATE_INCLUDES EXTERNAL_INCLUDES SOURCE_FILES HEADER_FILES LINK_LIBRARIES PRIVATE_LINK_LIBRARIES)
  ## make above arguments available as variables, e.g. ${openms_add_library_PRIVATE_LINK_LIBRARIES}
  cmake_parse_arguments(openms_add_library "${options}" "${oneValueArgs}" "${multiValueArgs}" ${ARGN} )

  #------------------------------------------------------------------------------
  # Status message for configure output
  message(STATUS "Adding library ${openms_add_library_TARGET_NAME}")

  #------------------------------------------------------------------------------
  # merge into global exported includes
  set(${openms_add_library_TARGET_NAME}_INCLUDE_DIRECTORIES ${openms_add_library_INTERNAL_INCLUDES}
                                                            ${openms_add_library_EXTERNAL_INCLUDES}
      CACHE INTERNAL "${openms_add_library_TARGET_NAME} include directories" FORCE)

  #------------------------------------------------------------------------------
  # Check if we want a unity build
  if (ENABLE_UNITYBUILD)
  	message(STATUS "Enabled Unity Build for ${openms_add_library_TARGET_NAME}")
  	convert_to_unity_build(${openms_add_library_TARGET_NAME}_UnityBuild openms_add_library_SOURCE_FILES)
  endif()

  #------------------------------------------------------------------------------
  # Add the library
  add_library(${openms_add_library_TARGET_NAME} ${openms_add_library_SOURCE_FILES})

  set_target_properties(${openms_add_library_TARGET_NAME} PROPERTIES CXX_VISIBILITY_PRESET hidden)
  set_target_properties(${openms_add_library_TARGET_NAME} PROPERTIES VISIBILITY_INLINES_HIDDEN 1)
  set_target_properties(${openms_add_library_TARGET_NAME} PROPERTIES AUTOMOC ON)

  #------------------------------------------------------------------------------
  # Include directories
  # since internal includes all start with include/OpenMS and install_headers takes care of merging them in the install tree,
  # we can reference them just by INSTALL_INCLUDE_DIR in the install tree. They are then included as usual via <OpenMS/OPENSWATHALGO/..>"
  target_include_directories(${openms_add_library_TARGET_NAME} PUBLIC
                             "$<BUILD_INTERFACE:${openms_add_library_INTERNAL_INCLUDES}>"
                             "$<INSTALL_INTERFACE:${INSTALL_INCLUDE_DIR}>"  # <prefix>/include
                             )

  # TODO actually we shouldn't need to add these external includes. They should propagate through target_link_library if they are public
  target_include_directories(${openms_add_library_TARGET_NAME} SYSTEM PUBLIC 
                             "$<BUILD_INTERFACE:${openms_add_library_EXTERNAL_INCLUDES}>"
                             "$<INSTALL_INTERFACE:${INSTALL_INCLUDE_DIR}>"
                             )
  target_include_directories(${openms_add_library_TARGET_NAME} SYSTEM PRIVATE ${openms_add_library_PRIVATE_INCLUDES})
  
  #TODO cxx_std_17 only requires a c++17 flag for the compiler. Not full standard support.
  # If we want full support, we need our own try_compiles (e.g. for structured bindings first available in GCC7)
  # or specify a min version of each compiler.
  target_compile_features(${openms_add_library_TARGET_NAME} PUBLIC cxx_std_20)

  # Add compiler flags using the new helper function
  openms_add_library_compiler_flags(${openms_add_library_TARGET_NAME})

  if(ADDRESS_SANITIZER)
    add_asan_to_target(${openms_add_library_TARGET_NAME})
  endif()
  
  
  set_target_properties(${openms_add_library_TARGET_NAME} PROPERTIES CXX_VISIBILITY_PRESET hidden)
  set_target_properties(${openms_add_library_TARGET_NAME} PROPERTIES VISIBILITY_INLINES_HIDDEN 1)

  #------------------------------------------------------------------------------
  # Generate export header if requested
  if(NOT ${openms_add_library_DLL_EXPORT_PATH} STREQUAL "")
    ## this snipped creates 'OpenMSConfig.h' in the build tree
    set(_CONFIG_H "include/${openms_add_library_DLL_EXPORT_PATH}${openms_add_library_TARGET_NAME}Config.h")
    string(TOUPPER ${openms_add_library_TARGET_NAME} _TARGET_UPPER_CASE)
    include(GenerateExportHeader)
    generate_export_header(${openms_add_library_TARGET_NAME}
                          EXPORT_MACRO_NAME ${_TARGET_UPPER_CASE}_DLLAPI
                          EXPORT_FILE_NAME ${_CONFIG_H})

    string(REGEX REPLACE "/" "\\\\" _fixed_path ${openms_add_library_DLL_EXPORT_PATH})

    # add generated header to visual studio
    source_group("Header Files\\${_fixed_path}" FILES ${_CONFIG_H})
  endif()

  #------------------------------------------------------------------------------
  # Link library against other libraries
  if(openms_add_library_LINK_LIBRARIES)
    ## check for consistent lib arch (e.g. all 64bit)?
    check_lib_architecture(openms_add_library_LINK_LIBRARIES)
    target_link_libraries(${openms_add_library_TARGET_NAME} PUBLIC ${openms_add_library_LINK_LIBRARIES})
    list(LENGTH openms_add_library_LINK_LIBRARIES _library_count)
  endif()

  if (openms_add_library_PRIVATE_LINK_LIBRARIES)
    ## check for consistent lib arch (e.g. all 64bit)?
    check_lib_architecture(openms_add_library_PRIVATE_LINK_LIBRARIES)
    target_link_libraries(${openms_add_library_TARGET_NAME} PRIVATE ${openms_add_library_PRIVATE_LINK_LIBRARIES})
    list(LENGTH openms_add_library_PRIVATE_LINK_LIBRARIES _private_library_count)
  endif()

  #------------------------------------------------------------------------------
  # Export libraries (self + dependencies)
  set(${openms_add_library_TARGET_NAME}_LIBRARIES
        ${openms_add_library_TARGET_NAME}
        ${openms_add_library_LINK_LIBRARIES}
        CACHE
        INTERNAL "${openms_add_library_TARGET_NAME} libraries" FORCE)

  #------------------------------------------------------------------------------
  # we also want to install the library
  install_library(${openms_add_library_TARGET_NAME})
  install_headers("${openms_add_library_HEADER_FILES};${PROJECT_BINARY_DIR}/${_CONFIG_H}" ${openms_add_library_TARGET_NAME})

  #------------------------------------------------------------------------------
  # register for export
  openms_register_export_target(${openms_add_library_TARGET_NAME})

  #------------------------------------------------------------------------------
  # On Windows copy DLLs and dependencies of them to other locations of executables that need them (tests, documenter)
  # TODO Find something that does not copy 100s of MB three times.
  # TODO I think this should ideally go to the tests and docs CMakeLists separately.
  # Copy target DLLs themselves
  copy_dll_to_extern_bin(${openms_add_library_TARGET_NAME})
  # Copy dependencies
  if(WIN32)
    # with newer CMakes we can also easily copy dependencies like Qt
    # This stores the command as a list
    set(has_dll_dep
            $<BOOL:$<TARGET_RUNTIME_DLLS:${openms_add_library_TARGET_NAME}>>
            )
    set(none_command
            ${CMAKE_COMMAND} -E echo
            )
    ## TODO check if we can use create_symlink instead
    set(copy_dlls_to_output_folder
            ${CMAKE_COMMAND} -E copy_if_different
            $<TARGET_RUNTIME_DLLS:${openms_add_library_TARGET_NAME}>
            $<TARGET_FILE_DIR:${openms_add_library_TARGET_NAME}>
            )

    if(GENERATOR_IS_MULTI_CONFIG)
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/src/tests/class_tests/bin/$<CONFIG>/" DLL_TEST_TARGET_PATH)
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/doc/doxygen/parameters/$<CONFIG>/" DLL_DOC_TARGET_PATH)
    else()
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/src/tests/class_tests/bin/" DLL_TEST_TARGET_PATH)
      file(TO_NATIVE_PATH "${OPENMS_HOST_BINARY_DIRECTORY}/doc/doxygen/parameters/" DLL_DOC_TARGET_PATH)
    endif()

    set(copy_dlls_to_test_folder
            ${CMAKE_COMMAND} -E copy_if_different
            $<TARGET_RUNTIME_DLLS:${openms_add_library_TARGET_NAME}>
            ${DLL_TEST_TARGET_PATH}
            )

    foreach(command IN ITEMS "${copy_dlls_to_output_folder}" "${copy_dlls_to_test_folder}")
      set(if_runtime_dlls_copy
              $<IF:${has_dll_dep},${command},${none_command}>
              )
      add_custom_command(TARGET ${openms_add_library_TARGET_NAME} POST_BUILD
              COMMAND "${if_runtime_dlls_copy}"
              COMMAND_EXPAND_LISTS
              )
    endforeach()

    if(ENABLE_DOCS)
        set(copy_dlls_to_doc_folder
         ${CMAKE_COMMAND} -E copy_if_different
         $<TARGET_RUNTIME_DLLS:${openms_add_library_TARGET_NAME}>
         ${DLL_DOC_TARGET_PATH}
         )
        set(if_runtime_dlls_copy
              $<IF:${has_dll_dep},${copy_dlls_to_doc_folder},${none_command}>
              )
        add_custom_command(TARGET ${openms_add_library_TARGET_NAME} POST_BUILD
              COMMAND ${CMAKE_COMMAND} -E make_directory "${DLL_DOC_TARGET_PATH}"
              COMMAND "${if_runtime_dlls_copy}"
              COMMAND_EXPAND_LISTS
              )
    endif()

    ## another fix for APPLE, see https://github.com/OpenMS/OpenMS/pull/7525
    if(APPLECLANG)
      if(CMAKE_CXX_COMPILER_VERSION VERSION_GREATER_EQUAL "15.0.0")
          target_link_options(${openms_add_library_TARGET_NAME} PRIVATE -ld_classic)
          set_target_properties(${openms_add_library_TARGET_NAME} PROPERTIES
              QT_NO_DISABLE_WARN_DUPLICATE_LIBRARIES TRUE)
      endif()
    endif()
  endif()
  #------------------------------------------------------------------------------
  # Status message for configure output
  message(STATUS "Adding library ${openms_add_library_TARGET_NAME} - SUCCESS")
endfunction()
