<!-- SPDX-FileCopyrightText: 2006-2023, Knut Reinert & Freie Universität Berlin -->
<!-- SPDX-FileCopyrightText: 2016-2023, Knut Reinert & MPI für molekulare Genetik -->
<!-- SPDX-License-Identifier: CC-BY-4.0 -->
# Tool Description Library (TDL)

## Introduction
This library offers a unified api to export tool description formats like [CTD](https://github.com/WorkflowConversion/CTDSchema) or [CWL](https://www.commonwl.org/).

The current CLI interfaces of [OpenMS](https://openms.de/) andi [SeqAn3](https://github.com/seqan/seqan3/) are very different. Using the same command line parsing library is currently (2022) not feasible for these projects.
While OpenMS already has support for CTD both are missing support for CWL. The idea is to replace the CTD exporter of OpenMS with the TDL implementation and integrating TDL into SeqAn3. As a next step TDL will gain support for CWL and with (hopefully) minimal code adjustments this
will give OpenMS and SeqAn3 based tools easy access to CWL tool description support.

## Usage (C++20)
At the core of TDL is the `ToolInfo` structure. It consist of three values:
```
struct ToolInfo {
    MetaInfo                meta{};
    std::vector<Node>       params{};
    std::vector<CLIMapping> cliMapping{};
};
```

Where `ToolInfo` carries
  - meta data about the tool
  - a tree of parameters `struct Node`
  - a tree of mapping from parameters to CLI prefixes.

The `Node` class is defined as:
 ```
struct Node {
    using Children = std::vector<Node>;
    using Value = std::variant<BoolValue,          // just a single bool value
                               IntValue,           // single int, double or string value
                               DoubleValue,
                               StringValue,
                               IntValueList,       // list of int, double or string values
                               DoubleValueList,
                               StringValueList,
                               Children>;          // not a value, but a node with children

    std::string name{};           //!< Name of the entry.
    std::string description{};    //!< Entry description.
    std::set<std::string> tags{}; //!< List of tags, e.g.: advanced parameter tag.
    Value value{Children{}};      //!< Current value of this entry
};

```

A CWL file is generated by calling `convertToCWL`
```
auto toolInfo = ToolInfo {
    ...
};
auto cwlAsString = convertToCWL(toolInfo);
std::cout << cwlAsString;
```

## Examples
- [Detailed Example](Example01.cpp.md)
- [Complete Example](Example00.cpp.md)

## Special cases
### CTD
  - string values with tags `{"input", "file"}`, `{"output", "file"}` or `{"output", "prefix"}`
    are exported as CTD typed values `input-file`, `output-file` or output-prefix`.
    The list valid string values are interpereted as `supported_formats`
  - list of string values with tags `{"input", "file"}` or `{"output", "file"}`
    are exported as CTD typed values `input-file` or `output-file`.
    The list valid string values are interpereted as `supported_formats`

### CWL
The following tags and tags combination have a special meaning
  - `{"file"}`: If used with `tdl::StringValue` a single input file, if used with `tdl::StringValueList` it represents a list of input files.
  - `{"directory"}`: Same as the tag `file` but with directories.
  - `{"file", "output"}`, A single or a list of input files (depending if `tdl::StringValue` or `tdl::StringValueList`)
  - `{"directory", "output"}`: Same as `{"file", "directory"}` but for directories.
  - `{"prefixed", "output"}`: A partial path, that is used as a prefix. If used with `tdl::StringValue` this signals only a single file will be created.
                              If used with `tdl::StringValueList` it signals that multiple files might be created.
  - `{"basecommand"}`: If used with `tdl::Node::Children` the value will be appended to the list of base-commands e.g.: `build` of `raptor build ...`
