// Copyright (c) 2002-present, OpenMS Inc. -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer: Axel Walter, Oliver Alka $
// $Authors: Axel Walter, Oliver Alka, Timo Sachsenberg $
// --------------------------------------------------------------------------

#include <OpenMS/ANALYSIS/ID/SiriusExportAlgorithm.h>
#include <OpenMS/APPLICATIONS/TOPPBase.h>
#include <OpenMS/DATASTRUCTURES/ListUtils.h>

using namespace OpenMS;

//----------------------------------------------------------
// Doxygen docu
//----------------------------------------------------------
/**
  @page TOPP_SiriusExport SiriusExport

  @brief De novo metabolite identification.

  Exports an input file for SIRIUS, a tool for metabolomics data analysis with several subtools, each serving specific purposes:

  - SIRIUS: Identify molecular formula for each compound individually using fragmentation trees and isotope patterns. Output from this tool can be used to generate an OpenSwathAssay library with the AssayGeneratorMetabo TOPP tool.
  - CSI:FingerID: This subtool is dedicated to predicting molecular structures based on tandem mass spectrometry (MS/MS) data. It utilizes a fragmentation tree approach for the annotation of fragment spectra.
  - CANOPUS: Predict compound categories for each compound individually based on its predicted molecular fingerprint (CSI:FingerID) using CANOPUS.
  - Passatutto: Compute decoy spectra based  on the fragmentation trees of the given input spectra. If no molecular formula is provided in the input, the top scoring computed formula is used. Required to include decoys in an OpenSwathWorkflow assay library generated by the AssayGeneratorMetabo TOPP tool.

  Sirius can be found at https://bio.informatik.uni-jena.de/software/sirius/

  <B>Internal procedure in SiriusExport</B>
  1. Input mzML (and optional featureXML) \n
  Make sure to input the matching mzML and featureXML files in the correct order.\n
  2. Preprocessing \n
  By providing a featureXML, the feature information can be used for feature mapping. \n
  Sirius will then process the internally merged MS2 spectra allocated to one feature (instead of all available MS2). \n
  To reduce the feature space even further a masstrace filter can be set. \n
  Additional adduct information can be provided using a featureXML from the MetaboliteAdductDecharger or AccurateMassSearch. \n
  3. Parsed by SiriusMSConverter into (sirius internal) .ms format
  
  <B>The command line parameters of this tool are:</B>
  @verbinclude TOPP_SiriusExport.cli
  <B>INI file documentation of this tool:</B>
  @htmlinclude TOPP_SiriusExport.html
*/

/// @cond TOPPCLASSES

class TOPPSiriusExport :
 public TOPPBase
{
 public:
  TOPPSiriusExport() :
    TOPPBase("SiriusExport", 
    "Metabolite identification using single and tandem mass spectrometry", true,
      {
        {"Kai Duehrkop and Sebastian Boecker",
         "Fragmentation trees reloaded",
         "J Cheminform; 2016",
         "10.1186/s13321-016-0116-8"},
        {"Kai Duehrkop, Huibin Shen, Marvin Meusel, Juho Rousu, and Sebastian Boecker",
         "Searching molecular structure databases with tandem mass spectra using CSI:FingerID",
         "Proceedings of the National Academy of Sciences; 2015",
         "10.1073/pnas.1509788112"}
      })
    {}

protected:

  void registerOptionsAndFlags_() override
  {
    registerInputFileList_("in", "<file(s)>", StringList(), "MzML Input file(s)");
    setValidFormats_("in", ListUtils::create<String>("mzML"));

    registerInputFileList_("in_featureinfo", "<file(s)>", StringList(), "FeatureXML input with feature and adduct information", false);
    setValidFormats_("in_featureinfo", ListUtils::create<String>("featureXML"));

    registerOutputFile_("out","<file>", "", "Internal SIRIUS .ms format after OpenMS preprocessing");
    setValidFormats_("out", ListUtils::create<String>("ms"));

    registerOutputFile_("out_compoundinfo","<file>", "", "File (.tsv) with information on processed compounds which are associated with a feature. Required for AssayGeneratorMetaboSirius tool.", false);
    setValidFormats_("out_compoundinfo", ListUtils::create<String>("tsv"));

    addEmptyLine_();

    registerFullParam_(SiriusExportAlgorithm().getDefaults());
  }

  ExitCodes main_(int, const char **) override
  {
    //-------------------------------------------------------------
    // Parsing parameters
    //-------------------------------------------------------------
    StringList mzML_files = getStringList_("in");
    StringList featureXML_files = getStringList_("in_featureinfo");
    String out_ms = getStringOption_("out");
    String out_compoundinfo = getStringOption_("out_compoundinfo");

    SiriusExportAlgorithm algorithm;     
    algorithm.setParameters(getParam_().copySubset(SiriusExportAlgorithm().getDefaults()));

    //-------------------------------------------------------------
    // Check input
    //-------------------------------------------------------------
    if (featureXML_files.size() > 0 && mzML_files.size() != featureXML_files.size())
    {
      throw Exception::MissingInformation(__FILE__, __LINE__, OPENMS_PRETTY_FUNCTION,
                                          "Number of .mzML do not match to the number of .featureXML files. \n Please check and provide the corresponding files.");
    }

    if (!out_compoundinfo.empty() && featureXML_files.size() == 0)
    {
      OPENMS_LOG_WARN << "A compound info output file was specified but no feature maps provided. The resulting table will be empty." << std::endl;
    }

    if (algorithm.isFeatureOnly() && featureXML_files.size() == 0)
    {
      throw Exception::IllegalArgument(__FILE__, __LINE__, OPENMS_PRETTY_FUNCTION,
                                           "No feature maps provided but preprocessing:feature_only flag set. Please provide featureXML files.");              
    }

    algorithm.run(mzML_files,
                  featureXML_files,
                  out_ms,
                  out_compoundinfo);

    return EXECUTION_OK;
  }
};

int main(int argc, const char ** argv)
{
  TOPPSiriusExport tool;
  return tool.main(argc, argv);
}

/// @endcond
