// Copyright (c) 2002-present, The OpenMS Team -- EKU Tuebingen, ETH Zurich, and FU Berlin
// SPDX-License-Identifier: BSD-3-Clause
//
// --------------------------------------------------------------------------
// $Maintainer:  $
// $Authors:  $
// --------------------------------------------------------------------------

#include <OpenMS/ANALYSIS/ID/PeptideSearchEngineFIAlgorithm.h>
#include <OpenMS/APPLICATIONS/TOPPBase.h>

#include <OpenMS/FORMAT/FileHandler.h>
#include <OpenMS/KERNEL/StandardTypes.h>
#include <OpenMS/SYSTEM/File.h>
#include <OpenMS/METADATA/PeptideIdentificationList.h>

using namespace OpenMS;
using namespace std;

//-------------------------------------------------------------
// Doxygen docu
//-------------------------------------------------------------

/**
    @page TOPP_PeptideDataBaseSearchFI PeptideDataBaseSearchFI

    @brief Identifies peptides in MS/MS spectra.

<CENTER>
    <table>
        <tr>
            <th ALIGN = "center"> pot. predecessor tools </td>
            <td VALIGN="middle" ROWSPAN=2> &rarr; PeptideDataBaseSearchFI &rarr;</td>
            <th ALIGN = "center"> pot. successor tools </td>
        </tr>
        <tr>
            <td VALIGN="middle" ALIGN = "center" ROWSPAN=1> any signal-/preprocessing tool @n (in mzML format)</td>
            <td VALIGN="middle" ALIGN = "center" ROWSPAN=1> @ref TOPP_IDFilter or @n any protein/peptide processing tool</td>
        </tr>
    </table>
</CENTER>

    @em This search engine is mainly for educational/benchmarking/prototyping use cases.
    It lacks behind in speed and/or quality of results when compared to state-of-the-art search engines.

    @note Currently mzIdentML (mzid) is not directly supported as an input/output format of this tool. Convert mzid files to/from idXML using @ref TOPP_IDFileConverter if necessary.
    @note Open-search mode is automatically determined by the precursor mass tolerance: enabled when tolerance exceeds 1 Da or 1000 ppm. No explicit open-search parameter is needed. This is logged at runtime and recorded in the output search parameters as UserParam 'open_search'.

    <B>The command line parameters of this tool are:</B>
    @verbinclude TOPP_PeptideDataBaseSearchFI.cli
    <B>INI file documentation of this tool:</B>
    @htmlinclude TOPP_PeptideDataBaseSearchFI.html
*/

// We do not want this class to show up in the docu:
/// @cond TOPPCLASSES

class PeptideDataBaseSearchFI :
    public TOPPBase
{
  public:
    PeptideDataBaseSearchFI() :
      TOPPBase("PeptideDataBaseSearchFI",
        "Annotates bottom-up MS/MS spectra using PeptideDataBaseSearchFI.",
        false)
    {
    }

  protected:
    void registerOptionsAndFlags_() override
    {
      registerInputFile_("in", "<file>", "", "input file ");
      setValidFormats_("in", ListUtils::create<String>("mzML"));

      registerInputFile_("database", "<file>", "", "input file ");
      setValidFormats_("database", ListUtils::create<String>("fasta"));

      registerOutputFile_("out", "<file>", "", "output file ");
      setValidFormats_("out", ListUtils::create<String>("idXML"));

      // put search algorithm parameters at Search: subtree of parameters
      Param search_algo_params_with_subsection;
      search_algo_params_with_subsection.insert("Search:", PeptideSearchEngineFIAlgorithm().getDefaults());
      registerFullParam_(search_algo_params_with_subsection);
    }

    ExitCodes main_(int, const char**) override
    {
      String in = getStringOption_("in");
      String database = getStringOption_("database");
      String out = getStringOption_("out");

      ProgressLogger progresslogger;
      progresslogger.setLogType(log_type_);

      vector<ProteinIdentification> protein_ids;
      PeptideIdentificationList peptide_ids;

      PeptideSearchEngineFIAlgorithm sse;
      sse.setParameters(getParam_().copy("Search:", true));
      // map algorithm exit code to application exit code
      PeptideSearchEngineFIAlgorithm::ExitCodes e = sse.search(in, database, protein_ids, peptide_ids);
      if (e != PeptideSearchEngineFIAlgorithm::ExitCodes::EXECUTION_OK)
      {
        return TOPPBase::ExitCodes::INTERNAL_ERROR;
      }

      // MS path already set in algorithm. Overwrite here so we get something testable
      if (getFlag_("test"))
      {
        // if test mode set, add file without path so we can compare it
        protein_ids[0].setPrimaryMSRunPath({"file://" + File::basename(in)});
      }

      FileHandler().storeIdentifications(out, protein_ids, peptide_ids, {FileTypes::IDXML});

      return EXECUTION_OK;
    }
};

int main(int argc, const char** argv)
{
  PeptideDataBaseSearchFI tool;
  return tool.main(argc, argv);
}

///@endcond
